/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/*
    FILEIO - Common file I/O functions
    Copyright (C) 1998  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/
/* $RCSfile: FILEIO.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	Wrapper functions of <stdio.h> functions.

	See fileio.h for any details.

   ----

   Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
   Known bugs:

   ----
	
*/

#include <assert.h>
#include <stdio.h>
#ifndef _MICROC_
#include <io.h>
#endif /*#	!defined(_MICROC_) */
#include <portable.h>

#ifdef __TURBOC__
#pragma hdrstop
#endif /*#	defined(__TURBOC__) */

#include "fileIO.h"
#include "yerror.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FILEIO.C 2.4 1998/10/04 05:17:36 ska Exp ska $";
#endif /*#	defined(RCS_Version) */


#ifdef _MICROC_
/*
 *	Bring a stream opened in Read & Write mode into another mode.
 *	This is necessary as Micro-C has trouble with seeking within
 *	R&W files.
 */
void setMode(FILE *fp, int mode)
{
	assert(fp);

	fflush(fp);
	switch(mode) {
	case TF_RD:	fp->FILE_options &= ~F_WRITE; break;
	default: fp->FILE_options |= F_WRITE; break;
	}
}
#endif /*#	defined(_MICROC_) */

/*
 *	Open a file
 *	On failure terminate with an error message
 */
FILE *Efopen(const char *fname, const char *mode)
{	FILE *fp;

	assert(fname);
	assert(mode);

	/* open file & emit error on failure */
	if((fp = fopen(fname, mode)) == NULL)
		fatal(E_openFile, fname);

	return fp;
}


/*
 *	Open a file with mode "wt", unless its first character is '+'.
 *	In that case open the file with mode "at".
 */
FILE *Efopen1(const char *fn)
{	iM(FILE *f;)

	assert(fn);

	if(*fn == '+') {
#ifndef _MICROC_
		return Efopen(fn + 1, appendMode);
#else /*#	!(!defined(_MICROC_)) */
	/* Micro-C has a problem, when the file does not already exist */
	if(f = fopen(fn + 1, appendMode))
		return f;

	++fn;			/* if failed -> use "wt" mode */
#endif /*#	!defined(_MICROC_) */
	}

	return Efopen(fn, "wt");
}


/*
 *	Close a file
 *	If an error occured while writing/reading the file, terminate
 *	with an error message.
 */
void Efclose(FILE *fp, const char *fnam)
{	/* ensure, fp->buffer is empty, so ferror() is up to date */

	assert(fp);
	assert(fnam);

	fflush(fp);

	if(ferror(fp))
		fatal(E_writeFile, fnam);

	fclose(fp);
}

/*
 *	Write into a file and terminate the program with an error message,
 *	if an error occurs.
 */
void Ewrite(void *buf, int size, FILE *fp, const char *fnam)
{
	assert(buf);
	assert(fp);
	assert(fnam);

#ifndef _MICROC_
	if(size && fwrite(buf, size, 1, fp) != 1)
#else /*#	!(!defined(_MICROC_)) */
	if(size && fput(buf, size, fp) != size)
#endif /*#	!defined(_MICROC_) */
		error(E_writeFile, fnam);
}

/*
 *	Read from a file and terminate the program with an error message,
 *	if an error occurs.
 */
void Eread(void *buf, int size, FILE *fp, const char *fnam)
{
	assert(buf);
	assert(fp);
	assert(fnam);
	
#ifndef _MICROC_
	if(size && fread(buf, size, 1, fp) != 1)
#else /*#	!(!defined(_MICROC_)) */
	if(size && fget(buf, size, fp) != size)
#endif /*#	!defined(_MICROC_) */
		error(E_readFile, fnam);
}

/*
 *	Seek the file with a _int_ offset
 */
void Eseek(int ofs, int whence, FILE *fp, const char *fnam)
{
	assert(fp);
	assert(fnam);

#ifdef _MICROC_
	if(fseek(fp, ofs < 0? -1: 0, ofs, whence))
#else /*#	!(defined(_MICROC_)) */
	if(fseek(fp, ofs, whence))
#endif /*#	defined(_MICROC_) */
		error(E_accessFile, fnam);
}

/*
 *	Seek to a specific position within the file
 */
#define Esetpos(pos,fp,fn) Eseekx(pos,0,fp,fn)
#define Eseekc(pos,fp,fn) Eseekx(pos,1,fp,fn)
void Eseekx(dword iM(*)pos, int whence, FILE *fp, const char *fnam)
{
	assert(fp);
	assert(fnam);
	
#ifdef _MICROC_
	if(fseek(fp, pos->hi, pos->lo, ofs, whence))
#else /*#	!(defined(_MICROC_)) */
	if(fseek(fp, pos, whence))
#endif /*#	defined(_MICROC_) */
		error(E_accessFile, fnam);
}

/*
 *	Aquire the current position within the file
 */
#define Egetpos(pos,fp,fnam) Etell(aS(pos), fp, fnam)
void Etell(dword nM(*)pos, FILE *fp, const char *fn)
{
	assert(fp);
	assert(fn);
	assert(pos);

#ifndef _MICROC_
	if((*pos = ftell(fp)) == -1)
#else /*#	!(!defined(_MICROC_)) */
	if(ftell(fp, &pos->hi, &pos->lo))
#endif /*#	!defined(_MICROC_) */
		error(E_accessFile, fn);
}

/*
 *	Read from the stream without changing the position
 *	terminate program on failure
 */
void Epeek(void *buf, int size, FILE *fp, const char *fnam)
{
	assert(fp);
	assert(fnam);
	assert(buf);
	
	if(size) {
#ifndef _MICROC_
		if(fread(buf, size, 1, fp) != 1)
#else /*#	!(!defined(_MICROC_)) */
		if(fget(buf, size, fp) != size)
#endif /*#	!defined(_MICROC_) */
			error(E_readFile, fnam);
		Eseek(-size, SEEK_CUR, fp, fnam);
	}
}


/*
 *	Check if a file exists
 */
#ifdef _MICROC_
int fileExists(const char *fnam)
{	int fd;

	assert(fnam);

	if((fd = open(fnam, F_READ)) <= 0)
		return 0;
	close(fd);
	return 1;
}

#endif /*#	defined(_MICROC_) */

