/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $Id: MSGCONF0.H 1.4 1998/10/07 04:42:53 ska Rel ska $
   $Locker: ska $	$Name:  $	$State: Rel $

	Global message handling definitions
		- part valid in all access methods:
			+ message interpreter
			+ message visulator

*/

#ifndef __MSGCONF0_H
#define __MSGCONF0_H

/* Maximum number of arguments skprintf() can process */
#define NL_ARGMAX	32



/*
 *	Issue an internal failure error message
 *	They have only a number to differ among themselves
 */
#define iFailure(nr) fatal(E_internalFailure, nr)

#define msgErrorNumber msgExitCode

/*
 *	What to do with ANSI escape sequences
 *	--> ANSI test flag
 */
#define ANSI_TEST 0	/* check if output stream is connected to a CON device */
#define ANSI_DENY 1	/* do no send ANSI sequences into the output stream */
#define ANSI_ALLOW 2	/* do not check for CON device, send sequences */

/*
 *	What to do with the noise levels
 *	--> noise level flags
 */
#define NOISE_TEST 2	/* check if stdin is redirected, before emitting
							interactive messages.
						   skip if not interactive message */
#define NOISE_DENY 1	/* skip messages of the noiselevel */
#define NOISE_ALLOW 0	/* emit messages of the noise level */
#define NOISE_ERR (-1)	/* error, returned by getNoiseLevel() on error */

/*
 *	Symbolic names for the noise levels
 */
#define ENoise_interactive 0	/* must be number zero !!! */
#define ENoise_informative 1
#define ENoise_fatal 2
#define ENoise_error 3
#define ENoise_warning 4
#define ENoise_none 5
#define ENoise_screen 6
#define	ENoise_Nr 7				/* number of noise levels */




void msgInitMI(void);
/* Initialize the message interpreter subsystems.

   	+ Disable VT100 features if no ANSI.SYS driver has been loaded

   This function must be called before skprintf().
   It is automatically called by msgInit().
*/

#define msgExitMI()
/* Shutdown the message interpreter subsystems.
	--> nothing
*/

void msgInitMV(void);
/* Initialize the message visulator subsystems.

   	+ Check if the stdin stream is redirected and disable interactive
   		messages
   	+ Determine the application's name (see appNameSave())

   This function must be called before error()/warning() etc. or
   appName() is called.
   It is automatically called by msgInit().
*/

#define msgExitMV()
/* Shutdown the message visulator subsystems.
	--> nothing
*/

void appNameSave(void);
/*	Initialize the appName() function
 */

/*
 *	Functions for the message visulator subsystem
 */

#ifdef _MICROC_
#define msgErrFct(fct) _msgErrFct(&(fct))
#define msgExitFct(fct) _msgErrFct(&(fct))
void _msgErrFct(int fct);
void _msgExitFct(int fct);
#else /*#	!(defined(_MICROC_)) */
void msgErrFct(void (*fct)(void));
void msgExitFct(void (*fct)(void));
#endif /*#	defined(_MICROC_) */
/* Set a function, which will be executed right before the exit()-call,
   after an error() or fatal() message has been emitted.
   msgExitFct() sets a function from within a msg?Init() function.
*/

void msgExit(void);
/*	Shutdown all message library subsystems
*/

char *appName(void);
/* Return the name of the current process. This name is upper-cased
   and without drive, path, and extension.
   See appNameSave().
*/

void setNoiseLevel(unsigned noiseLevel, int flag);
/* Function to set the allow/disallow flag for each noise level.

   May be called anytime the application wants to alter the interactive
   test behaviour of the interactive() function. To retrieve the current
   state of the flag use getNoiseLevel().

   flag should be one of NOISE_???, otherwise the behaviour of all
   functions using this flag is undefined.

   Initial setting is NOISE_ALLOW.

	For ENoise_interactive only:

   Should be called after msgInit() to ensure, that the test is
   performed anytime an interactive message is emitted.

   Initial setting of the ENoise_interactive flag: NOISE_TEST, but is
   altered to either NOISE_DENY or NOISE_ALLOW by the msgInit()
   function.

*/

int getNoiseLevel(unsigned noiseLevel);
/* Function retrieves the current state of the allow/disallow flag
   of the specified noise level.
*/



/* Message emitting functions:

   hlpScreen:
   	Emit the message E_hlpScreen onto stdout and exit with errorlevel.
   error:
   	Emit the supplied message onto stderr and exit with errorlevel.
   	Preceed the message with the application name and E_error.
   fatal:
   	Equals to error, but preceed with the application name and
   	E_fatal.
   warning:
   	Emit the supplied message onto stderr and preceed the message
   	with the application name and E_warning.
   informative:
   	Emit the supplied message onto stderr and preceed the message
   	with the application name.
   interactive:
   	Emit the supplied message onto stdout. If redirection test is 
   	enabled, the message is not emitted, if stdin is redirected to
   	a non-CON: device (or, by switch, to a file).
   smessage (screen):
   	Emit the supplied message onto stdout.
   message (none):
   	Emit the supplied message onto the supplied file stream.
*/

#ifdef _MICROC_
register void error(MSGID msg);
register void warning(MSGID msg); 
register void fatal(MSGID msg);
register void informative(MSGID msg);
register void interactive(MSGID msg);
register void smessage(MSGID msg);
register void message(FILE *fp, MSGID msg);

#else /*#	!(defined(_MICROC_)) */ /* no Micro-C */
void error(MSGID msg,...); 
void warning(MSGID msg,...); 
void fatal(MSGID msg,...);
void informative(MSGID msg,...);
void interactive(MSGID msg,...);
void smessage(MSGID msg,...);
void message(FILE *fp, MSGID msg,...);

#endif /*#	defined(_MICROC_) */

void hlpScreen(void);
	//DCL_MSG(E_hlpScreen);

/* no format string error reporting
#define Nr_E_hlpFmtStr 20
    DCL_MSG(E_hlpFmtStr);
*/




/*
 *	Functions for the message interpreter subsystem
 */

int putmc(int c, int rep, FILE *fout);
/* Dump the character rep times into the stream fout.
	This function is borrowed form the SUPPL library.

	Return:		 0: OK
				!0: failure
*/


int skprintf(unsigned char *poi, unsigned len, char *fmt, const unsigned *args);
/* input:	poi: output stream; must not be NULL!
   			len: length of the output buffer:
   				== 0 => output buffer is stream (poi is FILE*)
   				else => output buffer is memory (terminating NUL is added)
   					the buffer is filled with up to (len-1) bytes
   					is used to indicate, how much space is left in the buffer
   			fmt: enhanced format string
   			args: pointer to the first argument

   output: number of written characters or EOF on failure

   This function uses the isatty() function on the stream fout to
   determine, if to emit the ANSI enclosures or not. That behaviour is
   controlled by the ANSI test flag. When (len != 0) ANSI is always
   disabled.

*/

#ifndef _MICROC_
int fd_fprintf(FILE *fp, char *fmt, ...);
int fd_sprintf(char *buf, char *fmt, ...);
#else /*#	!(!defined(_MICROC_)) */
register int fd_fprintf(FILE *fp, char *fmt);
register int fd_sprintf(char *buf, char *fmt);
#endif /*#	!defined(_MICROC_) */
/* Wrapper for skprintf().

	These functions allow to call skprintf() with the same 
	calling interface than standard fprintf() and sprintf().
*/


void msgSetAnsiFlag(int flag);
/* Function to set the ANSI test flag. 

   Should be called before msgInit() to skip the check, if an ANSI
   driver is present. This test will set the flag to ANSI_TEST if a
   driver is present and to ANSI_DENY if none is found.

   In the functions using this flag, ANSI_TEST means, that ANSI will
   be used, if the file stream is connected to a console (CON:).

   May be called anytime the application wants to alter the ANSI test
   behaviour of skprintf(). To retrieve the current state of the flg
   use getAnsiTest().

   flag should be one of ANSI_????, otherwise the behaviour of all
   functions using this flag is undefined.

   Initial setting of the flag: ANSI_TEST.

*/

int msgGetAnsiFlag(void);
/* Function retrieves the current state of the ANSI test flag
*/

int checkANSI(void);
/* Return whether or not an ANSI driver is loaded.
*/



#endif /*#	!defined(__MSGCONF0_H) */
