#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <libgen.h>

#define OPT_FORCE   0x01
#define OPT_SLINK   0x02
#define OPT_LHELP   0x04
#define OPT_DTDIR   0x08
#define OPT_VBOSE   0x10
#define OPT_FPATH   0x20

static const char finishtext[] = "'.\n";
static const char betwentext[] = "' to `";
static const char missngtext[] = "Missing argument\n";
static const char createtext[] = "create symbolic link `";
static const char creathtext[] = "create hard link `";
static const char removetext[] = "Couldn't remove `";
static const char symfrmtext[] = "Couldn't create symlink from `";
static const char hrdfrmtext[] = "Couldn't create hard link from `";
static const char newlintext[] = "'\n";

static int flags = 0;

void writeerr( const char *s )
{
    _write( 2, s, strlen( s ) );
}

void writeout( const char *s )
{
    _write( 1, s, strlen( s ) );
}

int lastchar( char *s )
{
    return( s[ strlen( s ) - 1 ] );
}

int forcelink( const char *dest )
{
    if( flags & OPT_FORCE && _chmod( dest, 0 ) != -1 ) {
        if( unlink( dest ) == -1 ) {
            writeerr( removetext );
            writeerr( dest );
            writeerr( finishtext );
            return( 1 ); /* Failure */
        }
    }
    return( 0 );
}

void writeverbose( const char *dest, const char *source )
{
    if( flags & OPT_VBOSE ) {
        writeout( flags & OPT_SLINK ? createtext : creathtext );
        writeout( dest );
        writeout( betwentext );
        writeout( source );
        writeout( newlintext );
    }
}

int dolink( const char *source, const char *dest )
{
    if( flags & OPT_SLINK &&
         symlink( source, dest ) == -1 ) {
        writeerr( symfrmtext );
        writeerr( source );
        writeerr( betwentext );
        writeerr( dest );
        writeerr( finishtext );
        return( 1 );
    } else if( !( flags & OPT_SLINK ) &&
               link( source, dest ) == -1 ) {
        writeerr( hrdfrmtext );
        writeerr( source );
        writeerr( betwentext );
        writeerr( dest );
        writeerr( finishtext );
        return( 1 );
    }

    return( 0 );
}

int main( int argc, char **argv )
{
    int c;

    opterr = 0;

    while( ( c = getopt( argc, argv, "fsvp" ) ) != -1 ) {
        switch( c ) {
            case 'f':
                flags |= OPT_FORCE;
                break;
            case 's':
                flags |= OPT_SLINK;
                break;
            case 'v':
                flags |= OPT_VBOSE;
                break;
            case 'p':
                flags |= OPT_FPATH;
                break;
            case '?':
                flags |= OPT_LHELP;
                break;
        }
    }
    c = 0; /* We will use as the retval carrier today */

    if( flags & OPT_LHELP ) {
        writeerr( 
            "FreeDOS LN 2.0 - Create DJGPP-style hard and symbolic links\n" );
        writeerr(
            "Syntax: LN [-fsvp] <file1> <file2> [file3 ... destdir]\n" );
        writeerr(
            "Options:\n" );
        writeerr(
            "  -f  Force creation of link; overwrite destination files\n" );
        writeerr( 
            "  -s  Create symbolic links instead of hard links\n" );
        writeerr(
            "  -v  Write each file to screen before linking\n" );
        writeerr(
            "  -p  Use the full path of the source file[s]\n" );

        return( 1 );
    }

    if( ( c = _chmod( argv[ argc - 1 ], 0 ) ) & _A_SUBDIR && c != -1 ) {
        flags |= OPT_DTDIR;
    }

    if( !( flags & OPT_DTDIR ) ) {
        if( optind >= ( argc - 1 ) ) {
            writeerr( missngtext );
            c++;
        }

        for( ; optind < ( argc - 1 ); optind += 2 ) {
            char fullsource[ PATH_MAX ], fulldest[ PATH_MAX ];

            /* 
             * Always resolve the full dest path so as to resolve any given
             * symlinks - but leave the last component
             */
            if( realpath( dirname( argv[ optind + 1 ] ), fulldest ) == NULL )
                strcpy( fulldest, argv[ optind + 1 ] );
            else {
                strcat( fulldest, "\\" );
                strcat( fulldest, basename( argv[ optind + 1 ] ) );
            }

            if( !( flags & OPT_FPATH ) ||
                realpath( argv[ optind ], fullsource ) == NULL )
                strcpy( fullsource, argv[ optind ] );
            
            if( forcelink( fulldest ) ) {
                c++;
                continue;
            }
            
            writeverbose( fulldest, fullsource );

            if( dolink( fullsource, fulldest ) ) c++;
        }
    } else {
        char fullpath[ PATH_MAX ];
        char fullsource[ PATH_MAX ];

        argc--;

        if( optind >= argc ) {
            writeerr( missngtext );
            c++;
        }

        for( ; optind < argc; optind++ ) {
            strcpy( fullsource, argv[ argc ] );
            if( lastchar( fullsource ) != '/' &&
                lastchar( fullsource ) != '\\' )
                strcat( fullsource, "\\" );
            strcat( fullsource, argv[ optind ] );

            if( realpath( dirname( fullsource ), fullpath ) == NULL )
                strcpy( fullpath, fullsource );
            else {
                strcat( fullpath, "\\" );
                strcat( fullpath, basename( fullsource ) );
            }

            if( forcelink( fullpath ) ) {
                c++;
                continue;
            }

            if( !( flags & OPT_FPATH ) ||
                realpath( argv[ optind ], fullsource ) == NULL )
                strcpy( fullsource, argv[ optind ] );

            writeverbose( fullpath, fullsource );

            if( dolink( fullsource, fullpath ) ) c++;
        }
    }

    return( c );
}

