/**************************************************************************
*            Unix-like crypt(3) Algorithm for Password Encryption
*
*   File    : crypt3.c
*   Purpose : Provides crypt(3) functionality to ANSI C compilers
*             without a need for the crypt library.
*   Author  : Michael Dipperstein
*   Date    : November 3, 1998
*
***************************************************************************
*   I am releasing the source that I have provided into public domain
*   without any restrictions, warranties, or copyright claims of my own.
*
*   The code below has been cleaned and compiles correctly under, gcc,
*   lcc, and Borland's bcc C compilers.  A bug involving the left and
*   right halves of the encrypted data block in the widely published
*   crypt3.c source has been fixed by this version.  All implicit register
*   declarations have been removed, because they generated suboptimal code.
*   All constant data has been explicitly declared as const and all
*   declarations have been given a minimal scope, because I'm paranoid.
*
*   Caution: crypt() returns a pointer to static data.  I left it this way
*            to maintain backward compatibility.  The downside is that
*            successive calls will cause previous results to be lost.
*            This can easily be changed with only minor modifications to
*            the function crypt().
**************************************************************************/
#include <unistd.h>

/* Some definitions in setkey.c */
extern const char __IP[];
extern const char __FP[];
extern char __KS[ 16 ][ 48 ];
extern char __E[ 48 ];

/**************************************************************************
* Function:    crypt
*
* Description: Clone of Unix crypt(3) function.
*
* Inputs:      char *pw
*              pointer to 8 character encryption key (user password)
*              char *salt
*              pointer to 2 character salt used to modify the DES results.
*
* Returns:     Pointer to static array containing the salt concatenated
*              on to the encrypted results.  Same as stored in passwd file.
**************************************************************************/
char *crypt( char *pw, char *salt )
{
    int i, j, temp;
    char c,
         block[ 66 ];             /* 1st store key, then results */
    static char iobuf[ 16 ];      /* encrypted results */

    for( i = 0; i < 66; i++ )
        block[ i ] = 0;

    /* break pw into 64 bits */
    for( i = 0, c = *pw; c && ( i < 64 ); i++ )
    {
        for( j = 0; j < 7; j++, i++ )
            block[ i ] = ( c >> ( 6 - j ) ) & 01;
        pw++;
        c = *pw;
    }

    /* set key based on pw */
    setkey( block );

    for( i = 0; i < 66; i++ )
        block[ i ] = 0;

    for( i = 0; i < 2; i++ )
    {
        /* store salt at beginning of results */
        c = *salt++;
        iobuf[ i ] = c;

        if( c > 'Z' )
            c -= 6;

        if(c > '9')
            c -= 7;

        c -= '.';

        /* use salt to effect the E-bit selection */
        for( j = 0; j < 6; j++ )
        {
            if( ( c >> j ) & 01 )
            {
                temp = __E[ 6 * i + j ];
                __E[ 6 * i + j ] = __E[ 6 * i + j + 24 ];
                __E[ 6 * i + j + 24 ] = temp;
            }
        }
    }

    /* call DES encryption 25 times using pw as key and initial data = 0 */
    for( i = 0; i < 25; i++ )
        encrypt( block );

    /* format encrypted block for standard crypt(3) output */
    for( i = 0; i < 11; i++ )
    {
        c = 0;
        for( j = 0; j < 6; j++ )
        {
            c <<= 1;
            c |= block[ 6 * i + j ];
        }

        c += '.';
        if( c > '9' )
            c += 7;

        if( c > 'Z' )
            c += 6;

        iobuf[ i + 2 ] = c;
    }

    iobuf[ i + 2 ] = '\0';

    /* prevent premature NULL terminator */
    if( iobuf[ 1 ] == '\0' )
        iobuf[ 1 ] = iobuf[ 0 ];

    return( iobuf );
}

