#include <sys/stat.h>
#include <dirent.h>
#include <ftw.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static struct FTW __levels = { 0, 0 };

int nftw( char *directory,
          int( *funcptr )( const char *name,
                           const struct stat *ptr, int flag,
                           struct FTW *dinfo ),
          int depth, int flags )
{
    register DIR *  dirp;
    struct dirent * entp;
    struct stat     stats;
    register char * p;
    register int    i;
    long            seekpoint;
    char *          fullpath;

    /* If can't stat, tell the user so. */
    if( stat( directory, &stats ) < 0 ) 
        return( ( *funcptr )( directory, &stats, FTW_NS, &__levels ) );

    /* If it's not a directory, call the user's function. */
    if( ( stats.st_mode & S_IFMT ) != S_IFDIR )
       /* Saying "FTW_F" here is lying; what if this is a symlink? */
       return( ( *funcptr )( directory, &stats, FTW_F, &__levels ) );

    /* Open directory; if we can't, tell the user so. */
    dirp = opendir( directory );
    if( dirp == NULL )
        return( ( *funcptr )( directory, &stats, FTW_DNR, &__levels ) );

    /* See if user wants to go further. */
    if( !( flags & FTW_DEPTH ) ) {
        i = ( *funcptr )( directory, &stats, FTW_D, &__levels );
        if( i ) {
            closedir( dirp );
            return( i );
        }
    }

    __levels.level++;
    if( flags & FTW_CHDIR && chdir( directory ) < 0 ) return( -1 );

    /* Get ready to hold the full paths. */
    i = strlen( directory );
    fullpath = malloc( i + 1 + PATH_MAX + 1 );
    if( fullpath == NULL ) {
       closedir( dirp );
       return( -1 );
    }
    
    if( !( flags & FTW_CHDIR ) ) {
        strcpy( fullpath, directory );
        p = &fullpath[ i ];
        if( i && p[ -1 ] != '/' ) *p++ = '/';
    } else p = fullpath;

    /* Read all entries in the directory.. */
    while( entp = readdir( dirp ) )
       if( strcmp( entp->d_name, "." ) != 0 &&
           strcmp( entp->d_name, ".." ) != 0 ) {
           if( depth <= 1 ) {
               /* Going too deep; checkpoint and close this directory. */
               seekpoint = telldir( dirp );
               closedir( dirp );
               dirp = NULL;
           }

           /* Process the file. */
           strcpy( p, entp->d_name );
           i = nftw( fullpath, funcptr, depth - 1, flags );
           if( i ) {
               /* User's finished; clean up. */
               free( fullpath );
               if( dirp ) closedir( dirp );
               return( i );
           }

           /* Reopen the directory if necessary. */
           if( dirp == NULL ) {
               dirp = opendir( directory );
               if( dirp == NULL ) {
                   free( fullpath );
                   return( -1 );
               }
               seekdir( dirp, seekpoint );
           }
           __levels.base++;
       }

    if( flags & FTW_CHDIR && chdir( ".." ) < 0 ) return( -1 );
    __levels.level--;

    stat( directory, &stats );
/* If told to process directories after the fact, do it now */
    if( flags & FTW_DEPTH ) {
        i = ( *funcptr )( directory, &stats, FTW_D, &__levels );
        if( i ) {
            closedir( dirp );
            return( i );
        }
    }

    /* Clean up. */
    free( fullpath );
    closedir( dirp );
    return( 0 );
} 
