/*
 * An MPEG system stream demuxer.
 */

#include <string.h>
#include <stdio.h>

#include <allegro.h>

#include "apeg.h"
#include "mpeg1dec.h"

#ifdef _MSC_VER
#define strcasecmp stricmp
#endif

static int global_argc;
static const char **global_argv;

int check_parm(const char *param, int *idx)
{
	int i;

	for(i = global_argc-1;i >= 1;--i)
	{
		if(global_argv[i][0] == '-' && strcasecmp(global_argv[i]+1, param) == 0)
		{
			if(idx) *idx = i;
			return i;
		}
	}

	return 0;
}

int main(int argc, const char *argv[])
{
	PACKFILE *out_video, *out_audio, *in, *out;
	APEG_STREAM *stream;
	int done = FALSE;
	unsigned int i;
	char buf[2048], *ptr;

	if(allegro_init() != 0)
	{
		fprintf(stderr, "Error initializing Allegro: %s\n", allegro_error);
		return -1;
	}

	global_argc = argc;
	global_argv = argv;

	if(argc < 2 || check_parm("-help", NULL))
	{
		allegro_message("Usage: %s [options]\n"
		                "Available options:\n"
		                " [-infile <filename>] specifies the multiplexed filename to split\n"
		                "\n"
		                "Output filenames will be written to the current directory, using the input\n"
		                "file's name with the extension m1v (for video) and mp1/mp2/mp3/wav (for\n"
		                "audio).\n",
		                argv[0]);
		return 0;
	}

	fprintf(stderr, "Opening %s...", argv[1]);
	in = pack_fopen(argv[1], F_READ);
	if(!in)
	{
		fprintf(stderr, " unable to open!\n");
		return 1;
	}
	fprintf(stderr, " ok\n");

	ptr = get_filename(argv[1]);

	replace_extension(buf, ptr, "m1v", sizeof(buf));
	fprintf(stderr, "Creating %s...", buf);
	out_video = pack_fopen(buf, F_WRITE);
	if(!out_video)
		fprintf(stderr, " unable to open!\n");
	else
		fprintf(stderr, " ok\n");

	replace_extension(buf, ptr, "wav", sizeof(buf));
	fprintf(stderr, "Creating %s...", buf);
	out_audio = pack_fopen(buf, F_WRITE);
	if(!out_audio)
	{
		fprintf(stderr, " unable to open!\n");
		if(!out_video)
		{
			pack_fclose(in);
			return 1;
		}
	}
	else
		fprintf(stderr, " ok\n");

	fprintf(stderr, "Demuxing...");
	while(!done)
	{
		/* parse system layer, ignore everything we don't need */
		unsigned short len;
		unsigned int code;

		code = pack_mgetl(in);
		if(pack_feof(in))
			break;

		/* remove system layer byte stuffing */
		while((code & 0xFFFFFF00) != 0x100)
		{
			int c = pack_getc(in);
			if(c == EOF)
			{
				code = 0;
				break;
			}

			code = (code<<8) | c;
		}

		switch(code)
		{
			case PACK_START_CODE: /* pack header */
				/* skip pack header (system_clock_reference and mux_rate) */
				pack_fseek(in, 8);
				break;

			case VIDEO_ELEMENTARY_STREAM:
				out = out_video;
				goto write_it;

			case AUDIO_ELEMENTARY_STREAM:
				out = out_audio;
				goto write_it;

			write_it:
				len = pack_mgetw(in);

				code = pack_getc(in);
				len--;

				// parse MPEG-1 packet header
				while(code == 0xFF)
				{
					code = pack_getc(in);
					len--;
				}

				// stuffing bytes
				if(code >= 0x40)
				{
					if(code >= 0x80)
					{
						if(code != (unsigned)EOF)
							fprintf(stderr, " Error in packet header\n");
						done = TRUE;
						break;
					}

					// skip STD_buffer_scale
					pack_fseek(in, 1);
					code = pack_getc(in);
					len -= 2;

					if(code >= 0x40)
					{
						if(code != (unsigned)EOF)
							fprintf(stderr, " Error in packet header\n");
						done = TRUE;
						break;
					}
				}

				if(code >= 0x30)
				{
					len -= 9;
					pack_fseek(in, 9); // skip presentation and decoding time stamps
				}
				else if(code >= 0x20)
				{
					len -= 4;
					pack_fseek(in, 4); // skip presentation time stamps
				}
				else if(code != 0x0f)
				{
					if(code != (unsigned)EOF)
						fprintf(stderr, " Error in packet header\n");
					done = TRUE;
					break;
				}

				i = 0;
				while(i < len)
				{
					int bs = MIN(len-i, sizeof(buf));
					int s = pack_fread(buf, bs, in);
					if(s > 0)
						pack_fwrite(buf, s, out);

					if(s < bs)
					{
						done = TRUE;
						break;
					}

					i += s;
				}

				break;

			case ISO_END_CODE: /* end */
				done = TRUE;
				break;

			default:
				if(code >= SYSTEM_START_CODE)
				{
					/* skip system headers and non-video/audio packets */
					len = pack_mgetw(in);
					pack_fseek(in, len);
				}
//				else
//					fprintf(stderr, "Unexpected startcode 0x%08x in system layer", code);
		}
	}
	fprintf(stderr, " ok\n");

	pack_fclose(in);
	if(out_video)
		pack_fclose(out_video);
	if(out_audio)
	{
		pack_fclose(out_audio);

		// Open the audio file and determine the audio type (we don't want
		// to leave it as .wav if we don't have to)
		replace_extension(buf, ptr, "wav", sizeof(buf));
		fprintf(stderr, "Analyzing audio file...");
		stream = apeg_open_stream(buf);
		if(stream)
		{
			char fname[256];
			int flags = stream->flags;
			int layer = stream->audio.layer;

			apeg_close_stream(stream);

			strcpy(fname, buf);
			ptr = get_extension(fname);
			if(!ptr)
				return 0;

			if(flags & APEG_MPG_AUDIO)
			{
				sprintf(ptr, "mp%d", layer);
				if(rename(buf, fname) == 0)
					fprintf(stderr, "\n* renamed to %s\n", fname);
				else
					fprintf(stderr, "\n* could not change extension to '%s' (%s)\n", ptr, strerror(errno));
			}
			else
				fprintf(stderr, " unknown data type\n");
		}
		else
			fprintf(stderr, " unknown data type\n");
	}

	return 0;
}
END_OF_MAIN()
