\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Intel 8086 and 80386 Assembler and Meta Assembler 
\ CATEGORY    : Utilities 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : May 16, 1994, Coos Haak 
\ ----------------------------------------------------------------------


        MARKER -ASSEMBLER


DOC
   This is a full prefix 8086 assembler.
   With many extensions for 80386 code, when #cpu contains #386.
   Using of '$'-labeling as in F-PC possible.
   As are IF .. THEN structures etcetera.
   CODE <name> ... END-CODE  creates a code definition.
 ( LABEL <name>      ,,      creates a subroutine. )
   ;CODE                     is allowed.
ENDDOC

WARNING OFF
CASESENSITIVE ON
BASE @
HEX

\G Replace the first word list in the search order with the
\G ASSEMBLER word list.
VOCABULARY ASSEMBLER            ( -- )                  \ ASSEMBLER
ANS

ASSEMBLER DEFINITIONS ALSO

PRIVATES

\ Prefix system, like Intel's

CREATE APRIOR   PRIVATE 4 CELLS ALLOT
' DROP APRIOR !
' DROP APRIOR CELL+ CELL+ !

: A;!
        APRIOR CELL+ CELL+ 2!
        ;                               PRIVATE

: CLR-A;
        0 ['] DROP A;!
        ;                               PRIVATE

\G Terminate a line of assembly code.
: A;                    ( -- )                  \ ASSEMBLER
        APRIOR 2@ EXECUTE APRIOR CELL+ CELL+ 2@ APRIOR 2! CLR-A;
        ;

VARIABLE DST    PRIVATE
VARIABLE DMOD   PRIVATE
VARIABLE SRC    PRIVATE
VARIABLE SMOD   PRIVATE
VARIABLE DISP   PRIVATE
VARIABLE #VAL   PRIVATE
VARIABLE DIST   PRIVATE
VARIABLE ^ATAB  PRIVATE
VECTOR IMMED    PRIVATE
VARIABLE 1/2    PRIVATE
0 VALUE EXT?    PRIVATE

CREATE %REF     PRIVATE 2 CELLS ALLOT

1 CONSTANT WRD
2 CONSTANT REL

VARIABLE %PFA

: +PFA?
        %PFA C@ OR ;

: +PFA2?
        %PFA 1+ C@ OR ; PRIVATE

: %?REF
        %REF 1/2 @
        IF      CELL+
        THEN ;

VECTOR AHERE    ' HERE  IS AHERE
VECTOR AC@      ' C@    IS AC@
VECTOR AC!      ' C!    IS AC!
VECTOR AC,      ' C,    IS AC,
VECTOR A!       ' !     IS A!
VECTOR A,       ' ,     IS A,
VECTOR AALLOT   ' ALLOT IS AALLOT

: REF!
        AHERE SWAP ! DROP ;     PRIVATE

: RT,
        %REF @ ?DUP
        IF      REF! REL WRD + +PFA?
        THEN
        A, ;    PRIVATE

: RTC,
        %REF @ ?DUP
        IF      REF! REL +PFA?
        THEN
        AC, ;   PRIVATE

: AT,
        %REF @ ?DUP
        IF      REF! WRD +PFA?
        THEN
        A, ;    PRIVATE

: 1REF!
        AHERE %REF @ ! ;        PRIVATE

: 2REF!
        AHERE %REF CELL+ @ ! ;  PRIVATE

: FAR
        8 DIST ! ;

: INITASM
        %0100000000000110 DUP SRC ! DST ! SMOD OFF DMOD OFF DIST OFF
        %REF D0! %PFA OFF DISP OFF 1/2 OFF ['] NOOP IS IMMED
        CLEAR EXT?
        ;

: ?ASMERROR
        SWAP
        IF      >R
                0 ['] DROP APRIOR 2! CLR-A; INITASM -2 SET-ORDER DEFINITIONS
                R> THROW
        THEN
        DROP
        ;  PRIVATE

#-601 MESS" Operand or jump doesn't fit in a byte"
#-602 MESS" Wrong assembler control structure"
#-603 MESS" Invalid label"
#-604 MESS" Jump can't reach destiny"
#-605 MESS" No double labels allowed"
#-606 MESS" Error in labels"
#-607 MESS" No argument allowed"
#-608 MESS" Illegal or too less arguments used"
#-609 MESS" ESC number too large"
#-610 MESS" Immediate shift argument must be one"
#-611 MESS" Illegal far call or jump"

: ?BYTE
        DUP 80 -80 WITHIN #-601 ?ASMERROR
        ;

: ?PAIRS        ( x1 x2 -- )                    \ control numbers
        <> #-602 ?ASMERROR
        ;  PRIVATE

: >MARK         ( x -- addr x )                 \ mark forward address
        AHERE 0 AC,                                     \ mark source address
        SWAP                                            \ control number on top
        ;  PRIVATE

: >RESOLVE      ( addr x1 x2 -- )               \ resolve forward branch
        ?PAIRS                                          \ check control number
        AHERE 1- OVER - ?BYTE SWAP AC!                  \ resolve branch
        ;  PRIVATE

: <MARK         ( x -- addr x )                 \ mark backward branch
        AHERE                                           \ mark destiny address
        SWAP                                            \ control number on top
        ;  PRIVATE

: <RESOLVE      ( x -- )                        \ resolve backward branch
        ?PAIRS                                          \ check control number
        AHERE 1+ - ?BYTE AC,                            \ resolve branch
        ;  PRIVATE

-- Labels als in F-PC

20 CONSTANT MAXLABS     PRIVATE
 5 CONSTANT B/LAB       PRIVATE

VARIABLE USED?  PRIVATE

CREATE ^LABS    PRIVATE MAXLABS B/LAB * ALLOT

: LLAB-INIT
        ^LABS MAXLABS B/LAB * ERASE USED? OFF ; PRIVATE

LLAB-INIT

: LLAB>LINE
        DUP MAXLABS 1- U> #-603 ?ASMERROR
        B/LAB * ^LABS +
        ;  PRIVATE

\G Jump to an assembler label.
: $                     ( x -- )                        \ ASSEMBLER
        USED? ON LLAB>LINE 1 OVER CELL+ CELL+ C! DUP @
        IF      @
        ELSE    CELL+ DUP @ >R AHERE SWAP ! R> DUP 0=
                IF      AHERE +
                THEN
        THEN ;

: >RES
        CELL+ @
        0 OF    EXIT    THEN
        1+
        BEGIN   DUP AC@ >R AHERE OVER - 1- DUP 7F > #-604 ?ASMERROR
                OVER AC! R> DUP FE <>
        WHILE   FF00 OR + 2 +
        REPEAT
        2DROP
        ;  PRIVATE

: $:F
        USED? ON LLAB>LINE DUP @ 0<> #-605 ?ASMERROR
        DUP >RES AHERE SWAP !
        ;  PRIVATE

: ?RESOLVED
        USED? @
        IF      FALSE MAXLABS 0
                DO      I B/LAB * ^LABS + DUP CELL+ CELL+ C@ 0<>
                        IF      @ 0=
                                IF      CR
                                        ." Jump to undifined label "
                                        I . DROP TRUE
                                THEN
                        ELSE    @ 0<>
                                IF      CR ." Warning: label " I .
                                ." has no associated jumps to it"
                                THEN
                        THEN
                LOOP
                #-606 ?ASMERROR
        THEN ;

\G Define an assembler label.
: $:            ( x -- )                        \ ASSEMBLER
        ['] $:F A;! A; ;

\G Define a forward near label in assembler, one per definition.
: L$            ( -- addr )                     \ ASSEMBLER
        0 A; AHERE ;

\G Resolve a forward near label.
: L$:           ( addr -- )                     \ ASSEMBLER
        A; AHERE OVER - SWAP 2 - A! ;

: ?NOARG
        SRC @ %0100000000000110 <> DST @ %0100000000000110 <> OR
        #-607 ?ASMERROR
        ;  PRIVATE

: =SREG
        CREATE  C, ,
        DOES>   COUNT SMOD ! @ SRC ! ;  PRIVATE

: =DREG
        CREATE  C, ,
        DOES>   COUNT DMOD ! @ DST ! ;  PRIVATE

%1001010000000010 %11000000 =SREG AL
%1001010000000010 %11000000 =DREG AL,
%0001010000010010 %11000001 =SREG CL
%0001010000000010 %11000001 =DREG CL,
%0001010000000010 %11000010 =SREG DL
%0001010000000010 %11000010 =DREG DL,
%0001010000000010 %11000011 =SREG BL
%0001010000000010 %11000011 =DREG BL,
%0001010000000010 %11000100 =SREG AH
%0001010000000010 %11000100 =DREG AH,
%0001010000000010 %11000101 =SREG CH
%0001010000000010 %11000101 =DREG CH,
%0001010000000010 %11000110 =SREG DH
%0001010000000010 %11000110 =DREG DH,
%0001010000000010 %11000111 =SREG BH
%0001010000000010 %11000111 =DREG BH,

%1001010000000100 %11000000 =SREG AX
%1001010000000100 %11000000 =DREG AX,
%0001010000000100 %11000001 =SREG CX
%0001010000000100 %11000001 =DREG CX,
%0001010000100100 %11000010 =SREG DX
%0001010000100100 %11000010 =DREG DX,
%0001010000000100 %11000011 =SREG BX
%0001010000000100 %11000011 =DREG BX,
%0001010000000100 %11000100 =SREG SP
%0001010000000100 %11000100 =DREG SP,
%0001010000000100 %11000101 =SREG BP
%0001010000000100 %11000101 =DREG BP,
%0001010000000100 %11000110 =SREG SI
%0001010000000100 %11000110 =DREG SI,
%0001010000000100 %11000111 =SREG DI
%0001010000000100 %11000111 =DREG DI,

%0000001000000100 %11000000 =SREG ES
%0000001000000100 %11000000 =DREG ES,
%0000001000010100 %11000001 =SREG CS
%0000001000010100 %11000001 =DREG CS,
%0100001000000100 %11000010 =SREG SS
%0100001000000100 %11000010 =DREG SS,
%0000001000000100 %11000011 =SREG DS
%0000001000000100 %11000011 =DREG DS,

#CPU @ #386 = [IF]
%0000001000000100 %11000100 =SREG FS
%0000001000000100 %11000100 =DREG FS,
%0000001000000100 %11000101 =SREG GS
%0000001000000100 %11000101 =DREG GS,
[THEN]

: (#)
        SWAP DUP #VAL ! DUP 100 U< NEGATE 2* %0010000000000100 OR
        SWAP -80 80 WITHIN NEGATE OR ;  PRIVATE

: <#>
        (#) SRC ! IMMED ;       PRIVATE

: <#>,
        (#) DST ! IMMED ;       PRIVATE

\G Immediate mode for source.
: #             ( x -- x )                      \ ASSEMBLER
        APRIOR @ IS IMMED ['] <#> APRIOR ! ;

\G Immediate mode for destiny.
: #,            ( x -- x )                      \ ASSEMBLER
        APRIOR @ IS IMMED ['] <#>, APRIOR ! ;

\G Direct mode for source.
: []            ( -- )                          \ ASSEMBLER
        DISP ! 1/2 ON %0001100100000110 SRC ! %00000110 SMOD ! ;

\G Direct mode for destiny.
: [],           ( -- )                          \ ASSEMBLER
        DISP ! 1/2 ON %0001100100000110 DST ! %00000110 DMOD ! ;

: SrcInd
        SMOD ! %0001100000000110 SRC ! DUP DISP ! 1/2 ON ?DUP
        IF      80 -80 WITHIN NEGATE 1+ 6 LSHIFT
                SMOD @ %1111111100111111 AND OR SMOD !
        THEN ;  PRIVATE

: [Index]
        SMOD @ 7 AND
        CASE
                3 OF    %00000111       ENDOF
                5 OF    %01000110       ENDOF
                6 OF    %00000100       ENDOF
                %00000101 SWAP
        ENDCASE
        SrcInd ;        PRIVATE

: [Base+Index]
        SMOD @ 7 AND 3 - OR SrcInd ;    PRIVATE

: DstInd
        DMOD ! %0001100000000110 DST ! DUP DISP ! 1/2 ON ?DUP
        IF      80 -80 WITHIN NEGATE 1+ 6 LSHIFT
                DMOD @ %1111111100111111 AND OR DMOD !
        THEN ;  PRIVATE

: [Index],
        DMOD @ 7 AND
        CASE
                3 OF    %00000111       ENDOF
                5 OF    %01000110       ENDOF
                6 OF    %00000100       ENDOF
                %00000101 SWAP
        ENDCASE
        DstInd ;        PRIVATE

: [Base+Index],
        DMOD @ 7 AND 3 - OR DstInd ;    PRIVATE

: [BX]
        BX [Index] ;

: [BP]
        BP [Index] ;

: [SI]
        SI [Index] ;

: [DI]
        DI [Index] ;

: [BX+SI]
        BX 0 [Base+Index] ;

: [BX+DI]
        BX 1 [Base+Index] ;

: [BP+SI]
        BP 0 [Base+Index] ;

: [BP+DI]
        BP 1 [Base+Index] ;

: [BX],
        BX, [Index], ;

: [BP],
        BP, [Index], ;

: [SI],
        SI, [Index], ;

: [DI],
        DI, [Index], ;

: [BX+SI],
        BX, 0 [Base+Index], ;

: [BX+DI],
        BX, 1 [Base+Index], ;

: [BP+SI],
        BP, 0 [Base+Index], ;

: [BP+DI],
        BP, 1 [Base+Index], ;

: BYTE
        FFFB DUP SRC @ AND SRC ! DST @ AND DST ! ;

: MODE
        CREATE  , PRIVATE
        DOES>   C@ C, ; PRIVATE

: LINK
        HERE SWAP DUP
        IF      OVER - NEGATE
        THEN
        C, ;    PRIVATE

                04 MODE Aw      02 MODE Ab
16 MODE __
                24 MODE Dw      22 MODE Db      21 MODE Ds
                34 MODE Ew      32 MODE Eb
46 MODE Mw
                54 MODE Rw      52 MODE Rb
66 MODE Sw
76 MODE Xw
                A4 MODE Ow      B2 MODE Lb

CREATE [MATCH]  PRIVATE
        8000 , 4000 , 2000 , 1000 ,
        0800 , 0400 , 0200 , 0100 ,
        0080 , 0040 , 0020 , 0010 ,
        0008 ,

: (MATCH)
        2DUP AND 7 AND 0<> -ROT SWAP 3 RSHIFT 1E AND
        [MATCH] + @ AND 0<> AND ;       PRIVATE

: MATCH
        COUNT DST @ (MATCH) SWAP C@ SRC @ (MATCH) AND ; PRIVATE

: SIEVE
        BEGIN   DUP 1+ MATCH INVERT
        WHILE   DUP C@ DUP 0= #-608 ?ASMERROR
                -
        REPEAT
        3 + DUP 2 + ^ATAB ! @ EXECUTE INITASM ; PRIVATE

: <OPSYN>
        COUNT SWAP @ SIEVE ;    PRIVATE

: OPSYN
        CREATE  , PRIVATE
        DOES>   CREATE  SWAP C, @ ,
                DOES>   ['] <OPSYN> A;! A; ;    PRIVATE

: <OPCODE>
        @ SIEVE ;       PRIVATE

: OPCODE
        CREATE  ,
        DOES>   ['] <OPCODE> A;! A; ;   PRIVATE

: LTC,
        ^ATAB @ C@ AC, ;        PRIVATE

#CPU @ #386 = [IF]

: PFX,
        0F AC, ;        PRIVATE

[THEN]

: MODRM
        SWAP 7 AND 3 LSHIFT OVER OR AC, DUP C0 AND SWAP 6 = OVER 80 = OR
        IF      %REF @ 0<> DISP @ 0<> AND
                IF      1REF! WRD +PFA? A,
                ELSE    DISP @ A,
                THEN
        THEN
        40 =
        IF      %REF @ 0<> DISP @ 0<> AND
                IF      1REF! 0 +PFA? AC, EXIT
                THEN
                DISP @ AC,
        THEN ;  PRIVATE

: MDS
        DMOD @ SMOD @ MODRM ;   PRIVATE

: MSD
        SMOD @ DMOD @ MODRM ;   PRIVATE

: ??,
        IF      %REF CELL+ @
                IF      2REF! WRD +PFA2? A, EXIT
                THEN
                %REF @ 0<> DISP @ 0= AND
                IF      1REF! WRD +PFA? A, EXIT
                THEN
                #VAL @ A, EXIT
        THEN
        %REF CELL+ @
        IF      2REF! 0 +PFA2? AC, EXIT
        THEN
        %REF @ 0<> DISP @ 0= AND
        IF      1REF! 0 +PFA? AC, EXIT
        THEN
        #VAL @ AC, ;    PRIVATE

: ARI.1
        LTC, DMOD @ MODRM ^ATAB @ C@ 81 = ??, ; PRIVATE

: ARI.1D
        SWAP [], ARI.1 ;        PRIVATE

: ARIT
        3 LSHIFT ^ATAB @ C@ OR AC, ;    PRIVATE

: ARI.2
        ARIT #VAL @ A, ;        PRIVATE

: ARI.3
        ARIT #VAL @ AC, ;       PRIVATE

: ARI.4
        ARIT MSD ;      PRIVATE

: ARI.4D
        SWAP [], ARI.4 ;        PRIVATE

: ARI.5
        ARIT MDS ;      PRIVATE

: ARI.5S
        SWAP [] ARI.5 ; PRIVATE

0       LINK __ Dw ' ARI.1D , 81 C,
        LINK __ Ds ' ARI.1D , 83 C,
        LINK __ Rw ' ARI.4D , 1 C,
        LINK Rw __ ' ARI.5S , 3 C,
        LINK Eb Db ' ARI.1 , 80 C,
        LINK Ew Dw ' ARI.1 , 81 C,
        LINK Ew Ds ' ARI.1 , 83 C,
        LINK Aw Dw ' ARI.2 ,    5 C,
        LINK Ab Db ' ARI.3 ,    4 C,
        LINK Eb Rb ' ARI.4 ,    0 C,
        LINK Ew Rw ' ARI.4 ,    1 C,
        LINK Rb Eb ' ARI.5 ,    2 C,
        LINK Rw Ew ' ARI.5 ,    3 C, OPSYN ARITH

: DI.1
        3 LSHIFT 40 OR SMOD @ 7 AND OR AC, ;    PRIVATE

: DI.2
        LTC, SMOD @ MODRM ;     PRIVATE

: DI.2S
        SWAP [] DI.2 ;  PRIVATE

0       LINK __ __ ' DI.2S , FF C,
        LINK __ Eb ' DI.2 , FE C,
        LINK __ Ew ' DI.2 , FF C,
        LINK __ Rw ' DI.1 ,             OPSYN DINC

: PP.1
        3 LSHIFT 40 OR 20 XOR SMOD @ 7 AND OR AC, ;     PRIVATE

#CPU @ #386 = [IF]

: PP.2
        DUP 7 = SRC @ 10 AND AND #-608 ?ASMERROR
        SMOD @ 3 AND 3 LSHIFT SMOD @ 4 AND
        IF      PFX, NIP ^ATAB @ C@
        THEN
        OR AC, ;        PRIVATE

[ELSE]

: PP.2
        DUP 7 = SRC @ 10 AND AND #-608 ?ASMERROR
        SMOD @ 3 AND 3 LSHIFT OR AC, ;  PRIVATE

[THEN]

: PP.3
        7 OF    0       THEN
        DUP
        IF      FF
        ELSE    8F
        THEN
        AC, SMOD @ MODRM ;      PRIVATE

: PP.3S
        SWAP [] PP.3 ;  PRIVATE

0       LINK __ __ ' PP.3S ,
        LINK __ Ew ' PP.3 ,
        LINK __ Sw ' PP.2 , A1 C,
        LINK __ Rw ' PP.1 , OPSYN POP1

#CPU @ #386 = [IF]

: PP.4
        DROP LTC, #VAL @ A, ;   PRIVATE

: PP.5
        DROP LTC, #VAL @ AC, ;  PRIVATE

0       LINK __ __ ' PP.3S ,
        LINK __ Ew ' PP.3 ,
        LINK __ Sw ' PP.2 , A0 C,
        LINK __ Rw ' PP.1 ,
        LINK __ Dw ' PP.4 , 68 C,
        LINK __ Ds ' PP.5 , 6A C, OPSYN PUSH1

: BT.1
        PFX, BA AC, 4 OR DMOD @ MODRM #VAL @ 1F AND AC, ;       PRIVATE

: BT.1D
        SWAP [], BT.1 ; PRIVATE

: BT.2
        PFX, 3 LSHIFT A3 OR AC, MSD ;   PRIVATE

: BT.2D
        SWAP [], BT.2 ; PRIVATE

0       LINK __ Db ' BT.1D ,
        LINK __ Rw ' BT.2D ,
        LINK Ew Db ' BT.1 ,
        LINK Ew Rw ' BT.2 , OPSYN BT1

: SET.1
        PFX, AC, 0 SMOD @ MODRM ;       PRIVATE

: SET.1S
        SWAP [] SET.1 ; PRIVATE

0       LINK __ __ ' SET.1S ,
        LINK __ Eb ' SET.1  , OPSYN SETCC

: MVX.1
        PFX, AC, MDS ;  PRIVATE

: MVX.2
        PFX, 1 + AC, MDS ;      PRIVATE

: MVX.1S
        SWAP [] MVX.1 ; PRIVATE

0       LINK Rw __ ' MVX.1S ,
        LINK Rw Ew ' MVX.2  ,
        LINK Rw Eb ' MVX.1  , OPSYN MOVX

: BSF.1
        PFX, AC, MDS ;  PRIVATE

: BSF.1S
        SWAP [] MVX.1 ; PRIVATE

0       LINK Rw __ ' BSF.1S ,
        LINK Rw Ew ' BSF.1  , OPSYN BSF/R

[THEN]

: <LOOP>
        ?NOARG C@ AC, AHERE 1+ - ?BYTE RTC, INITASM ;   PRIVATE

: LOOPING
        CREATE  C,
        DOES>   ['] <LOOP> A;! A; ;     PRIVATE

#CPU @ #386 = [IF]

: <COND>
        ?NOARG C@ SWAP AHERE 2 + - DUP -80 80 WITHIN
        IF      SWAP AC, RTC, INITASM EXIT
        THEN
        PFX, SWAP 10 + AC, 2 - RT, INITASM ;    PRIVATE

: COND
        CREATE  C,
        DOES>   ['] <COND> A;! A; ;     PRIVATE

[ELSE]

' LOOPING ALIAS COND    PRIVATE

[THEN]

: MOV.1
        LTC, 0 DMOD @ MODRM ^ATAB @ C@ 1 AND ??, ;      PRIVATE

: MOV.1D
        [], MOV.1 ;     PRIVATE

: MOV.3
        DMOD @ 7 AND B0 OR AC, #VAL @ AC, ;     PRIVATE

: MOV.4
        DMOD @ 7 AND B8 OR AC, #VAL @ A, ;      PRIVATE

: MOV.5
        LTC, MSD ;      PRIVATE

: MOV.5D
        [], MOV.5 ;     PRIVATE

: MOV.6
        LTC, MDS ;      PRIVATE

: MOV.6S
        [] MOV.6 ;      PRIVATE

: MOV.7
        LTC, DISP @ AT, ;       PRIVATE

: MOV.7S
        [] MOV.7 ;      PRIVATE

: MOV.7D
        [], MOV.7 ;     PRIVATE

: XCG.1
        DMOD @ SMOD @ OR 7 AND 90 OR AC, ;      PRIVATE

: TST.3
        A8 AC, #VAL @ AC, ;     PRIVATE

: TST.4
        A9 AC, #VAL @ A, ;      PRIVATE

: RET.1
        C3 DIST @ OR AC, ;      PRIVATE

: RET.2
        C2 DIST @ OR AC, #VAL @ A, ;    PRIVATE

: ENT.1
        C8 AC, #VAL @ A, 0 AC, ;        PRIVATE

: ESC.1
        #VAL @ 40 U< INVERT #-609 ?ASMERROR
        #VAL @ 3 RSHIFT D8 OR AC,
        #VAL @ SMOD @ MODRM ;   PRIVATE

: IO.1
        LTC, #VAL @ AC, ;       PRIVATE

: AAM.1
        D4 OR AC, #VAL @ AC, ;  PRIVATE

: AAM.2
        D4 OR AC, 0A AC, ;      PRIVATE

0       LINK __ Db ' AAM.1 ,
        LINK __ __ ' AAM.2 , OPSYN AAMER

: <SINGLE>
        ?NOARG C@ AC, INITASM ; PRIVATE

: SINGLE
        CREATE  C,
        DOES>   ['] <SINGLE> A;! A; ;   PRIVATE

: OVERRIDE
        CREATE  C,
        DOES>   C@ AC, ;        PRIVATE

: M/D.1
        LTC, SMOD @ MODRM ;     PRIVATE

: M/D.1S
        SWAP [] M/D.1 ; PRIVATE

0       LINK __ __ ' M/D.1S , F7 C,
        LINK __ Eb ' M/D.1  , F6 C,
        LINK __ Ew ' M/D.1  , F7 C, OPSYN M/D

#CPU @ #386 = [IF]

: IMUL.1
        LTC, 5 SMOD @ MODRM ;   PRIVATE

: IMUL.1S
        [] IMUL.1 ;     PRIVATE

: IMUL.2
        PFX, AF AC, MDS ;       PRIVATE

: IMUL.2S
        [] IMUL.2 ;     PRIVATE

: IMUL.3
        DUP -80 80 WITHIN
        IF      6B AC, MDS AC, EXIT
        THEN
        69 AC, MDS A, ; PRIVATE

: IMUL.3S
        SWAP [] IMUL.3 ;        PRIVATE

[THEN]

: INT.1
        3 OF    CC AC, EXIT
        THEN
        0CD AC, AC, ;   PRIVATE

#CPU @ #386 = [IF]

: SFT.1
        #VAL @ 1-
        IF      ^ATAB INCR LTC, DMOD @ MODRM #VAL @ AC, EXIT
        THEN
        LTC, DMOD @ MODRM ;     PRIVATE

[ELSE]

: SFT.1
        #VAL @ 1- #-610 ?ASMERROR
        LTC, DMOD @ MODRM ;     PRIVATE

[THEN]

: SFT.1D
        SWAP [], SFT.1 ;        PRIVATE

: SFT.2
        LTC, DMOD @ MODRM ;     PRIVATE

: SFT.2D
        SWAP [], SFT.2 ;        PRIVATE

#CPU @ #386 = [IF]

0       LINK __ Db ' SFT.1D , D1 C, C1 C,
        LINK __ Lb ' SFT.2D , D3 C,
        LINK Eb Db ' SFT.1 , D0 C, C0 C,
        LINK Ew Db ' SFT.1 , D1 C, C1 C,
        LINK Eb Lb ' SFT.2 , D2 C,
        LINK Ew Lb ' SFT.2 , D3 C, OPSYN SFT

[ELSE]

0       LINK __ Db ' SFT.1D , D1 C,
        LINK __ Lb ' SFT.2D , D3 C,
        LINK Eb Db ' SFT.1 , D0 C,
        LINK Ew Db ' SFT.1 , D1 C,
        LINK Eb Lb ' SFT.2 , D2 C,
        LINK Ew Lb ' SFT.2 , D3 C, OPSYN SFT

[THEN]

: CAL.1
        0FF AC, DIST @ 0<> NEGATE ^ATAB @ C@ OR SMOD @ MODRM ;  PRIVATE

: CAL.2
        DIST @ #-611 ?ASMERROR
        CAL.1
        ;  PRIVATE

: CAL.3
        DIST @
        IF      9A AC, A, A, EXIT
        THEN
        E8 AC, AHERE 2 + - RT, ;        PRIVATE

: JMP.3
        DIST @
        IF      EA AC, A, A, EXIT
        THEN
        AHERE 2 + - DUP -80 80 WITHIN %REF @ 0= AND
        IF      EB AC, RTC, EXIT
        THEN
        E9 AC, 1- RT, ; PRIVATE

: LDS.1
        C4 OR AC, MDS ; PRIVATE

: LDS.1S
        SWAP [] LDS.1 ; PRIVATE

0       LINK Rw __ ' LDS.1S ,
        LINK Rw Ew ' LDS.1 , OPSYN L.DS

#CPU @ #386 = [IF]

: LFS.1
        PFX, B0 OR AC, MDS ;    PRIVATE

: LFS.1S
        SWAP [] LFS.1 ; PRIVATE

0       LINK Rw __ ' LFS.1S ,
        LINK Rw Ew ' LFS.1 , OPSYN L.FS

[THEN]

0       LINK __ __ ' A, ,   OPCODE DW   PRIVATE

0       LINK __ __ ' AC, ,  OPCODE DB   PRIVATE

: CCON
        CREATE  C,
        DOES>   C@ ;    PRIVATE

0 ARITH ADD     1 ARITH OR      2 ARITH ADC     3 ARITH SBB
4 ARITH AND     5 ARITH SUB     6 ARITH XOR     7 ARITH CMP

0 DINC INC      1 DINC DEC

7 POP1 POP

#CPU @ #386 = [IF]

6 PUSH1 PUSH

[ELSE]

6 POP1 PUSH

[THEN]

70 COND JO
71 COND JNO
72 COND JB
73 COND JAE
74 COND JZ
75 COND JNZ
76 COND JBE
77 COND JA
78 COND JS
79 COND JNS
7A COND JPE
7B COND JPO
7C COND JL
7D COND JGE
7E COND JLE
7F COND JG

E0 LOOPING LOOPNZ
E1 LOOPING LOOPZ
E2 LOOPING LOOP
E3 LOOPING JCXZ

0       LINK __ Dw ' MOV.1D , C7 C,
        LINK __ Rw ' MOV.5D , 89 C,
        LINK Rw __ ' MOV.6S , 8B C,
        LINK __ Sw ' MOV.5D , 8C C,
        LINK Sw __ ' MOV.6S , 8E C,
        LINK Aw __ ' MOV.7S , A1 C,
        LINK __ Aw ' MOV.7D , A3 C,
        LINK Eb Db ' MOV.1 , C6 C,
        LINK Ew Dw ' MOV.1 , C7 C,
        LINK Rb Db ' MOV.3 ,
        LINK Rw Dw ' MOV.4 ,
        LINK Eb Rb ' MOV.5 , 88 C,
        LINK Ew Rw ' MOV.5 , 89 C,
        LINK Rb Eb ' MOV.6 , 8A C,
        LINK Rw Ew ' MOV.6 , 8B C,
        LINK Ew Sw ' MOV.5 , 8C C,
        LINK Sw Ew ' MOV.6 , 8E C,
        LINK Ab Xw ' MOV.7 , A0 C,
        LINK Aw Xw ' MOV.7 , A1 C,
        LINK Xw Ab ' MOV.7 , A2 C,
        LINK Xw Aw ' MOV.7 , A3 C, OPCODE MOV

0       LINK Rw __ ' MOV.6S , 8D C,
        LINK Rw Mw ' MOV.6 , 8D C, OPCODE LEA

0       LINK __ Rw ' MOV.5D , 87 C,
        LINK Rw __ ' MOV.6S , 87 C,
        LINK Eb Rb ' MOV.5 , 86 C,
        LINK Ew Rw ' MOV.5 , 87 C,
        LINK Rb Eb ' MOV.6 , 86 C,
        LINK Rw Ew ' MOV.6 , 87 C,
        LINK Rw Aw ' XCG.1 ,
        LINK Aw Rw ' XCG.1 ,            OPCODE XCHG

0       LINK __ Dw ' MOV.1D , F7 C,
        LINK __ Rw ' MOV.5D , 85 C,
        LINK Rw __ ' MOV.6S , 85 C,
        LINK Eb Db ' MOV.1 , F6 C,
        LINK Ew Dw ' MOV.1 , F7 C,
        LINK Ab Db ' TST.3 ,
        LINK Aw Dw ' TST.4 ,
        LINK Eb Rb ' MOV.5 , 84 C,
        LINK Ew Rw ' MOV.5 , 85 C,
        LINK Rb Eb ' MOV.6 , 84 C,
        LINK Rw Ew ' MOV.6 , 85 C, OPCODE TEST

0       LINK __ __ ' RET.1 ,
        LINK __ Dw ' RET.2 , OPCODE RET

0       LINK Db Mw ' ESC.1 , OPCODE ESC

#CPU @ #386 = [IF]

0       LINK __ Dw ' ENT.1 , OPCODE ENTER

[THEN]

0       LINK Ab Db ' IO.1 , E4 C,
        LINK Aw Db ' IO.1 , E5 C,
        LINK Ab Ow ' LTC, , EC C,
        LINK Aw Ow ' LTC, , ED C, OPCODE IN

0       LINK Db Ab ' IO.1 , E6 C,
        LINK Db Aw ' IO.1 , E7 C,
        LINK Ow Ab ' LTC, , EE C,
        LINK Ow Aw ' LTC, , EF C, OPCODE OUT

0 AAMER AAM     1 AAMER AAD

27 SINGLE DAA           2F SINGLE DAS
37 SINGLE AAA           3F SINGLE AAS
90 SINGLE NOP
98 SINGLE CBW           99 SINGLE CWD
9B SINGLE WAIT
9C SINGLE PUSHF         9D SINGLE POPF
9E SINGLE SAHF          9F SINGLE LAHF
A4 SINGLE MOVSB         A5 SINGLE MOVSW
A6 SINGLE CMPSB         A7 SINGLE CMPSW
AA SINGLE STOSB         AB SINGLE STOSW
AC SINGLE LODSB         AD SINGLE LODSW
AE SINGLE SCASB         AF SINGLE SCASW
CE SINGLE INTO          CF SINGLE IRET
F0 SINGLE LOCK
F2 SINGLE REPNZ         F3 SINGLE REPZ      ' REPZ ALIAS REP
F4 SINGLE HLT
F5 SINGLE CMC
F8 SINGLE CLC           F9 SINGLE STC
FA SINGLE CLI           FB SINGLE STI
FC SINGLE CLD           FD SINGLE STD
D7 SINGLE XLAT

#CPU @ #386 = [IF]

60 SINGLE PUSHA         61 SINGLE POPA
6C SINGLE INSB          6D SINGLE INSW
6E SINGLE OUTSB         6F SINGLE OUTSW
C9 SINGLE LEAVE

[THEN]

26 OVERRIDE ES: 2E OVERRIDE CS: 36 OVERRIDE SS: 3E OVERRIDE DS:

#CPU @ #386 = [IF]

64 OVERRIDE FS: 65 OVERRIDE GS: \ 66 OVERRIDE SZ: \ 67 OVERRIDE AD:

: SZ:
        EXT? TRUE TO EXT?
        IF      EXIT
        THEN
        66 AC,
        ;

: EAX
        SZ: AX ;

: EBX
        SZ: BX ;

: ECX
        SZ: CX ;

: EDX
        SZ: DX ;

: ESI
        SZ: SI ;

: EDI
        SZ: DI ;

: EAX,
        SZ: AX, ;

: EBX,
        SZ: BX, ;

: ECX,
        SZ: CX, ;

: EDX,
        SZ: DX, ;

: ESI,
        SZ: SI, ;

: EDI,
        SZ: DI, ;

: CWDE
        CBW SZ: ;

: CDQ
        CWD SZ: ;

0 BT1 BT        1 BT1 BTS       2 BT1 BTR               3 BT1 BTC

90 SETCC SETO
91 SETCC SETNO
92 SETCC SETB
93 SETCC SETAE
94 SETCC SETZ
95 SETCC SETNZ
96 SETCC SETBE
97 SETCC SETA
98 SETCC SETS
99 SETCC SETNS
9A SETCC SETPE
9B SETCC SETPO
9C SETCC SETL
9D SETCC SETGE
9E SETCC SETLE
9F SETCC SETG

B6 MOVX MOVZX
BE MOVX MOVSX

BC BSF/R BSF
BD BSF/R BSR

[THEN]

2 M/D NOT       3 M/D NEG       4 M/D MUL       6 M/D DIV       7 M/D IDIV

#CPU @ #386 = [IF]

0       LINK __ __ ' IMUL.1S , F7 C,
        LINK Rw __ ' IMUL.2S ,
        LINK __ Eb ' IMUL.1 , F6 C,
        LINK __ Ew ' IMUL.1 , F7 C,
        LINK Rw Ew ' IMUL.2 ,                   OPCODE IMUL

0       LINK Rw __ ' IMUL.3S ,
        LINK Rw Ew ' IMUL.3 , OPCODE IMUL#

[ELSE]

5 M/D IMUL

[THEN]

0       LINK __ __ ' INT.1 , OPCODE INT

0 SFT ROL       1 SFT ROR       2 SFT RCL       3 SFT RCR
4 SFT SHL       5 SFT SHR       ' SHL ALIAS SAL 7 SFT SAR

0       LINK __ Ew ' CAL.1 , 2 C,
        LINK __ Rw ' CAL.2 , 2 C,
        LINK __ __ ' CAL.3 ,            OPCODE CALL

0       LINK __ Ew ' CAL.1 , 4 C,
        LINK __ Rw ' CAL.2 , 4 C,
        LINK __ __ ' JMP.3 ,            OPCODE JMP

0 L.DS LES      1 L.DS LDS

#CPU @ #386 = [IF]

2 L.FS LSS      4 L.FS LFS      5 L.FS LGS

[THEN]

?DEF CCON [IF]

70 CCON NOV     71 CCON OV
72 CCON U>=     73 CCON U<
74 CCON 0<>     75 CCON 0=
76 CCON U>      77 CCON U<=
78 CCON 0>=     79 CCON 0<
7A CCON PO      7B CCON PE
7C CCON >=      7D CCON <
7E CCON >       7F CCON <=
E3 CCON CNZ

EB CCON NEVER   PRIVATE

: IF
        >R A; R> AC, 11 >MARK ;

: AHEAD
        NEVER IF ;

: THEN
        A; 11 >RESOLVE ;

: ENTRY
        1 CS-ROLL THEN ;

: ELSE
        AHEAD ENTRY ;

: BEGIN
        A; 12 <MARK ;

: UNTIL
        >R A; R> AC, 12 <RESOLVE ;

: AGAIN
        NEVER UNTIL ;

: REPEAT
        AGAIN THEN ;

: WHILE
        IF 1 CS-ROLL ;

: DO
        A; AHERE ;

\ Same definitions as in FORTH, but not IMMEDIATE or COMPILE-ONLY
: CS-PICK
        >R A; R> CS-PICK
        ;  ANS

: CS-ROLL
        >R A; R> CS-ROLL
        ;  ANS

[THEN]

-- Macros

\G Assemble "ccc" as an 8 bit value.
' DB ALIAS DB       ( "ccc" -- )                \ ASSEMBLER

\G Assemble "ccc" as a 16 bit value.
' DW ALIAS DW       ( "ccc" -- )                \ ASSEMBLER

: NEXT
                LODSW   ES:
                JMP     AX ;

: $ODD
        A; AHERE 1 [ FORTH ] AND 0=
        IF      [ ASSEMBLER ] CLD [ FORTH ]
        THEN
        [ ASSEMBLER ] ;

: $EVEN
        A; AHERE 1 [ FORTH ] AND
        IF      [ ASSEMBLER ] CLD [ FORTH ]
        THEN
        [ ASSEMBLER ] ;

: >ASM
        [ FORTH ] ALSO ASSEMBLER INITASM LLAB-INIT CLR-A;
        ;

FORTH DEFINITIONS

\G Skip leading space delimiters. Parse name delimited by a space.
\G Create a definition for name, called a "code definition", with
\G the execution semantics defined below. Add the ASSEMBLER word
\G list to the search order and start interpreting the rest of the
\G parse area and assemble machine code. If needed, refill the input
\G buffer until END-CODE is processed.
\G
\G name Execution: ( i*x -- j*x )
\G Execute the machine code sequence that was generated following
\G CODE .
\G See also: END-CODE
: CODE                  ( "name" -- )                   \ ASSEMBLER
        BL PARSE-WORD HEAD, >ASM
        ;  ANS

\ : LABEL
\       GET-CURRENT ['] ASSEMBLER >BODY SET-CURRENT
\       CREATE SET-CURRENT >ASM ;

\G Interpretation: ( i*x -- )
\G This word is marked compile only. The default interpreter issues
\G exception -14 when an attempt is made to execute this word.
\G
\G Compilation: ( C: colon-sys -- )
\G Append the execution semantics defined below to the current
\G definition. End the current definition, consuming colon-sys,
\G enter interpret state, add the ASSEMBLER word list to the search
\G order and start interpreting the rest of the parse area and
\G assemble machine code. If needed, refill the input buffer until
\G END-CODE is processed.
\G
\G Execution: ( -- ) ( R: nest-sys -- )
\G Replace the execution semantics of the most recently defined word
\G with the name execution semantics given below. Return control to
\G the calling definition specified by nest-sys. An ambiguous
\G condition exists if the most recently defined word was not
\G defined with CREATE or a user-defined word that calls CREATE .
\G
\G name Execution: ( i*x -- j*x )
\G Perform the machine code sequence that was generated following
\G ;CODE .
\G See also: DOERCODE DOES> END-CODE
: ;CODE                                                     \ ASSEMBLER
        [ INTERNAL ] POSTPONE MODIFY [ FORTH ]
        HERE L, 0 , 0 ,
        POSTPONE [ >ASM [ INTERNAL ] CLEAR LOCALS [ FORTH ]
        ;  IMMEDIATE  COMPILE-ONLY  ANS

EXTRA DEFINITIONS

\G Skip leading space delimiters. Parse name delimited by a space.
\G Create a definition for name with the execution semantics defined
\G below. Enter interpret state, add the ASSEMBLER word list to the
\G search order and start interpreting the rest of the parse area
\G and assemble machine code. If needed, refill the input buffer
\G until END-CODE is processed.
\G
\G Execution: ( -- ) ( R: nest-sys -- )
\G Replace the execution semantics of the most recently defined word
\G with the name execution semantics given below. Return control to
\G the calling definition specified by nest-sys. An ambiguous
\G condition exists if the most recently defined word was not
\G defined with CREATE or a user-defined word that calls CREATE .
\G
\G name Execution: ( i*x -- j*x )
\G Perform the machine code sequence that was generated following
\G DOERCODE .
\G See also: DOES> END-CODE
: DOERCODE              ( "name" -- )                   \ ASSEMBLER
        : POSTPONE ;CODE COMPILE-ONLY
        ;

FORTH DEFINITIONS

\G Resolve all assembler labels, terminate the current code
\G definition and allow its name to be found in the dictionary.
\G Remove the ASSEMBLER word list from the search order.
: END-CODE              ( -- )                          \ ASSEMBLER
        A; ?RESOLVED PREVIOUS REVEAL
        ;

ASSEMBLER DEFINITIONS

\G Terminate a $IF386 directive.
: $THEN                 ( -- )                          \ ASSEMBLER
        A; ;

\G Jump to after $THEN .
: $ELSE                 ( -- )                          \ ASSEMBLER
        A; [ FORTH ]
        BEGIN   SCAN-ANY ['] $THEN =
        UNTIL
        ;

\G If #CPU does not contain 386 jump to after $ELSE or $THEN .
\G Else continue.
: $IF386                ( -- )                          \ ASSEMBLER
        A; [ FORTH ] #CPU @ #386 <>
        IF      BEGIN   SCAN-ANY
                        ['] $THEN OF    EXIT    THEN
                        ['] $ELSE =
                UNTIL
        THEN
        ;

DEPRIVE

PREVIOUS FORTH DEFINITIONS

WARNING ON
CASESENSITIVE OFF
BASE !

                            \ (* End of Source *) /
