\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Utilities for a better environment 
\ CATEGORY    : Kernel extensions 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : May 04, 1994, Coos Haak 
\ ----------------------------------------------------------------------



\ some other comment prefixes

' \ ALIAS \G immediate  ( "ccc<eol>" -- )               \ EXTRA
\G If BLK contains zero, parse and discard the remainder of the
\G parse area; otherwise parse and discard the portion of the parse
\G area corresponding to the remainder of the current line. \G is an
\G immediate word. Used in generating glossaries.

\G If BLK contains zero, parse and discard the remainder of the
\G parse area; otherwise parse and discard the portion of the parse
\G area corresponding to the remainder of the current line. -- is an
\G immediate word.
' \ ALIAS -- immediate  ( "ccc<eol>" -- )               \ EXTRA

\ Date and time routines

extra definitions

privates

\G Array of three letter month names. Months in normal order, but
\G letters reversed. To be used in pictured number strings.
\G See also: .SHORTDATE
CREATE MONTHS   ( -- c-addr )                           \ EXTRA
        #36 allot                                   \ in reverse order for HOLD
s" najbeframrpayamnujlujguapestcovonced" months swap cmove

\G c-addr u specify a character string containing the date in the
\G format "dd mmm yy".
\G See also: (DATE) .SHORTDATE
: (SHORTDATE)           ( -- c-addr u )             \ EXTRA "paren-shortdate"
        push base decimal                               \ save base, decimal
        date <# u>d # # bl hold 2drop                   \ convert year
        1- 3 * months +                                 \ get month
        count hold count hold c@ hold bl hold           \ add month
        u>d # # #>                                      \ add day
        pop base                                        \ restore base
        ;

\G Display the date in the format "dd mmm yy".
\G See also: (SHORTDATE) MONTHS
: .SHORTDATE            ( -- )                          \ EXTRA "dot-shortdate"
        (shortdate) type space
        ;

here ", rebmeceD"                       \ compile the names of the months
here ", rebmevoN"                       \ in reverse order for HOLD
here ", rebotcO"
here ", rebmetpeS"
here ", tsuguA"
here ", yluJ"
here ", enuJ"
here ", yaM"
here ", lirpA"
here ", hcraM"
here ", yraurbeF"
here ", yraunaJ"

create mnames                           \ array with pointers to the names
        , , , , , , , , , , , ,
        private

\G c-addr u specify a character string containing the date in the
\G format "Month day, year".
\G See also: .DATE
: (DATE)                ( -- c-addr u )                 \ EXTRA "paren-date"
        push base decimal                               \ save base, decimal
        date <# u>d # # # # 2drop bl hold ',' hold      \ convert year
        swap u>d # # bl hold rot                        \ convert day
        1- mnames []cell @ count 0                      \ get address of month
        do      count hold                              \ put name in string
        loop
        drop #>                                         \ ready with string
        pop base                                        \ restore base
        ;

\G Display the date in the format "Month day, year".
\G See also: (DATE) .SHORTDATE .TIME
: .DATE                 ( -- )                          \ EXTRA "dot-date"
        (date) type space
        ;

: :##                   ( x -- )                        \ convert two digits
        u>d #                                           \ second digit 0..9
        6 base ! # ':' hold decimal 2drop               \ first digit 0..5
        ;  private

\G c-addr u specify a character string containing the time in the
\G format "hh:mm:ss".
\G See also: .TIME
: (TIME)                ( -- c-addr u )                 \ EXTRA "paren-time"
        push base decimal                               \ save base, decimal
        time swap rot <# :## :## u>d # # #>             \ three pairs of digits
        pop base                                        \ restore base
        ;

\G Display the time in the format "hh:mm:ss".
\G See also: (TIME) .DATE
: .TIME                 ( -- )                          \ EXTRA "dot-time"
        (time) type space
        ;

deprive

forth definitions

\ some key definitions

' bye $1000 []key !                                     \ set ALT-Q

:noname                 ( -- )          \ press ALT-X to leave with a message
                                        \ for DOS
        1 halt                                          \ set 1 in retcode
        ;  $2D00 []key !                                \ set ALT-X

\ some conditional routines

extra definitions

\G Skip leading space delimiters. Parse name delimited by a space.
\G Find name. If name is found, flag is true, false otherwise.
\G See also: ?UNDEF
: ?DEF                  ( "name" -- flag )              \ EXTRA "query-defined"
        bl word find nip 0<>                            \ leave a flag
        ;

\G Skip leading space delimiters. Parse name delimited by a space.
\G Find name. If name is found, flag is false, true otherwise.
\G See also: ?DEF
: ?UNDEF                ( "name" -- flag )          \ EXTRA "query-undefined"
        ?def invert                                     \ invert the flag
        ;

\ timing and diagnose

\G c-addr u specify a string containing the time elapsed since the
\G last execution of TIMER-RESET in the format of a numeric string
\G with three digits after the decimal point.
\G See also: .ELAPSED
: (.T0)                 ( -- c-addr u )             \ EXTRA "paren-dot-t-zero"
        push base decimal                               \ save base, decimal
        gettime timesave 2@ d-                          \ calculate millisecs
        <# # # # '.' hold #s #>                         \ make string
        pop base                                        \ restore base
        ;

\G Display the elapsed time as specified by (.T0) followed by the
\G string " seconds elapsed.".
\G See also: (.T0) .MS TIMER-RESET
: .ELAPSED      ( -- )                                  \ EXTRA "dot-elapsed"
        (.t0) type ."  seconds elapsed."                \ since timer-reset
        ;

\G Display the elapsed time as specified by (.T0) followed by the
\G string " seconds.".
\G See also: (.T0) .ELAPSED TIMER-RESET
: .MS           ( -- )                                  \ EXTRA "dot-m-s"
        (.t0) type ."  seconds."
        ;

:noname                 ( -- )          \ give some diagnostics since start
        push base decimal                               \ save base, decimal
        timesave 2@                                     \ move timer out of
        timesave 2 cells + 2@ timesave 2!               \ the way
        gettime timesave 2@ d-                          \ elapsed millisecs
        #10 um/mod 1 max >r drop                        \ centisecs, zero div!
        cr here bytes @ - dup 5 u.r ." +" u>d           \ code bytes compiled
        lhere lbytes @ - dup 5 u.r ." +" m+             \ list bytes
        hhere hbytes @ - dup 5 u.r m+                   \ header bytes
        ." =" 2dup 6 ud.r space                         \ the sum of these
        #100 r@ m*/ 5 ud.r ."  bps,"                    \ how many per second
        linesread @ 6 u.r ."  lines,"                   \ lines read
        linesread @ u>d #6000 r> m*/
        <# # # # ',' hold #s #> 9 over - spaces type
        ."  lpm, "                                      \ how many per minute
        (.t0) type space                                \ the time difference
        timesave 2!                                     \ restore normal timer
        pop base                                        \ restore base
        ;  is diagnose                                  \ put in vector

forth definitions

\ routines for text modes

internal also extra definitions

privates

\G Reset the display to the same textmode as at startup.
\G See also: TEXT0 TEXT?
: TEXT                  ( -- )                          \ EXTRA
        ['] (prompt) is prompt dftmode normal true to text?
        ;

: other-textmodes
        ['] (prompt) is prompt setmode normal true to text?
        ;  private

\G Set the display to 80 x 25 color text mode.
\G See also: TEXT TEXT0 TEXT1 TEXT2 TEXT?
: TEXT0         ( -- )                                  \ EXTRA "text-zero"
        3 other-textmodes
        ;                               \ you may have to change 3 into 7 or 2

\G Set the display to 132 x 25 text mode. Only available with
\G Speedstar Pro ?
\G See also: TEXT0 TEXT2
: TEXT1         ( -- )                                  \ EXTRA "text-one"
        $14 other-textmodes
        ;                                       \ change for your own monitor

\G Set the display to 132 x 43 text mode. Only available with
\G Speedstar Pro ?
\G See also: TEXT0 TEXT1
: TEXT2         ( -- )                                  \ EXTRA "text-two"
        $54 other-textmodes
        ;                                       \ change for your own monitor

:noname
        ( chain atexit )                        \ Vector is NOOP now
        text? invert                            \ No text mode ?
        if      text                            \ Text mode
        then
    ;  is atexit                                \ Set vector

deprive

previous forth definitions

\ dumps

extra definitions

\G Display the contents of u consecutive addresses starting at
\G extended address x-addr. At the beginning of the line the
\G extended address is displayed, followed with the hexadecimal
\G contents of 16 characters and then the same characters are
\G displayed with SEMIT .
\G
\G DUMPX is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
\G See also: DUMP
: DUMPX         ( x-addr u -- )                         \ EXTRA "dump-extended"
        rot local dumpseg                               \ save the segment
        c/l #10 - #17 / 1 max 4 * local width           \ setup for screen
        over + 2dup paragraph-aligned swap $-10 and     \ start a $XXX0
        ?do     cr dumpseg i x.                         \ type address
                i width + i
                do      i 3 and 0=                      \ in columns of four
                        if      space
                        then
                        i i 2over within                \ in range?
                        if      dumpseg swap c@x b.     \ type it
                        else    drop 3 spaces           \ not in range
                        then
                loop
                space i width + i
                do      i i 2over within                \ in range?
                        if      dumpseg swap c@x        \ get char
                        else    drop bl                 \ space
                        then
                        semit
                loop
                stop? ?leave                            \ escape leaves
        width +loop                                     \ next round
        2drop                                           \ drop range
        ;

forth definitions

\G Display the contents of u consecutive addresses starting at addr.
\G At the beginning of the line the address is displayed, preceded
\G with the name of the segment, followed with the hexadecimal
\G contents of 16 characters and then the same characters are
\G displayed with SEMIT .
\G
\G DUMP is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
: DUMP          ( addr u -- )                           \ FORTH
        cseg -rot dumpx                                 \ extend address
        ;  ans

extra definitions

\G Display the contents of u consecutive addresses starting at
\G list address l-addr. At the beginning of the line the extended
\G address is displayed, followed with the hexadecimal contents of
\G 16 characters and then the same characters are displayed with
\G SEMIT .
\G
\G LDUMP is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
\G See also: DUMP DUMPX LTYPE
: LDUMP         ( l-addr u -- )                         \ EXTRA "list-dump"
        lseg -rot dumpx                                 \ extend address
        ;

\G Display the contents of u consecutive addresses starting at
\G header address x-addr. At the beginning of the line the extended
\G address is displayed, followed with the hexadecimal contents of
\G 16 characters and then the same characters are displayed with
\G SEMIT .
\G
\G HDUMP is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
\G See also: DUMP DUMPX HTYPE
: HDUMP         ( h-addr u -- )                         \ EXTRA "head-dump"
        hseg -rot dumpx                                 \ extend address
        ;

\G If u is greater than zero, display the character string at the
\G list address l-addr for a total of u characters. The characters
\G are displayed as with SEMIT .
\G See also: HTYPE LDUMP
: LTYPE         ( l-addr u -- )                         \ EXTRA "list-type"
        lseg -rot stypex                                \ extend address
        ;

\G If u is greater than zero, display the character string at the
\G header address h-addr for a total of u characters. The characters
\G are displayed as with SEMIT .
\G See also: HDUMP LTYPE
: HTYPE         ( h-addr u -- )                         \ EXTRA "head-type"
        hseg -rot stypex                                \ extend address
        ;

forth definitions

\G Copy and display the values currently on the data stack. Starting
\G on a new line, a '(' (left parenthesis) followed by a space is
\G displayed. Then follow the values on the stack, when BASE
\G contains 10, as signed numbers, unsigned otherwise. At the end a
\G ')' (right parenthesis) is displayed.
\G
\G .S is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
: .S            ( -- )                                  \ FORTH "dot-s"
        depth 0< -4 ?error                              \ stack underflow
        cr ." ( " depth ?dup
        if      0 swap 1-
                do      i pick from base #10 =
                        if      .               \ only decimal with sign
                        else    u.              \ other bases unsigned
                        then
                -1 +loop
        then
        ." ) "
        ;  ans

extra definitions

\G Display n as a signed decimal number.
\G See also: .HEX
: .DEC          ( n -- )                                \ EXTRA "dot-decimal"
        push base decimal
        .
        pop base
        ;

\G a-addr is the address of a cell containing the current value of
\G indentation for the decompiler.
VARIABLE INDENT         ( -- a-addr )                   \ EXTRA
         indent off              \ pretty printing

\G Increment the current value of the indentation with eight.
: IND++         ( -- )                              \ EXTRA "indent-increment"
        8 indent +! ;

\G Decrement the current value of the indentation with eight.
: IND--         ( -- )                              \ EXTRA "indent-decrement"
        indent @ 8 - 8 max indent ! ;

\G Display a new line and emit the number of spaces contained in
\G INDENT .
: NL            ( -- )                                  \ EXTRA "new-line"
        out indent @ =
        if      exit
        then
        cr indent @ spaces ;

\G Increment INDENT with eight and perform NL .
: >NL           ( -- )                              \ EXTRA "indent-forward"
        ind++ nl ;

\G Decrement INDENT with eight and perform NL .
: <NL           ( -- )                              \ EXTRA "indent-backward"
        ind-- nl ;

privates

variable #times         private
        1 #times !

\G Execute the text before on the same line repeatedly for n times.
\G See also: MANY
: TIMES         ( n -- )                                \ EXTRA
        #times incr #times @ <
        if      1 #times ! exit
        then
        >in off ;                                       \ reinterpret the line

deprive

\G Execute the text before on the same line repeatedly until a
\G keypress.
\G See also: TIMES
: MANY          ( -- )                                  \ EXTRA
        stop? invert
        if      >in off                                 \ reinterpret the line
        then ;

:noname
        chain start                                     \ extend chain
        getmode 7 =                                     \ mono or hercules
        if      1 $B000
        else    $12 $B800
        then
        ['] sbase >body ! statusattr !
        get-directory throw current-directory place     \ save directory name
        ;  is start

\G a-addr is the address of a cell containing the delay after WORDS
\G SEE DIS etc. in millseconds.
VARIABLE WORDSPEED      ( -- addr )                     \ EXTRA
      10 wordspeed !

\G If the word list identification wid has a header (when it was
\G created with VOCABULARY ), return a true flag else a false flag.
: REGULAR?      ( wid -- wid flag )                     \ EXTRA "regular-query"
        dup body> >head head>name
        h@ split drop 0<>
        ;

\G If the length of the name associated with the dictionary entry
\G address dea does not fit on the current line, perform a CR . Type
\G the name and wait for the time in milliseconds contained in
\G WORDSPEED .
: .HEAD         ( dea -- )                              \ EXTRA "dot-head"
        (.head) dup out + c/l < invert              \ does it fit on the line
        if      cr                                      \ no, new line
        then
        type                                            \ type name
        wordspeed @ ms                                  \ wait for some time
        ;

\G Display the name of the word list identification wid.
\G See also: .HEAD
: .VOCNAME      ( wid -- )                              \ EXTRA "dot-vocname"
        body> >head .head
        ;

get only set-current

\G Display the word lists in the search order in their search order
\G sequence, from the first searched to the last searched. Also
\G display the word list into which new definitions will be placed.
\G
\G ORDER is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
: ORDER         ( -- )                                  \ FORTH
        cr ." Context: " get-order 0
        ?do     .vocname space space
        loop
        cr ." Current: " get-current .vocname
        ;  ans

\ Escape from '0 set-order': alt-o
:noname
        order -2 set-order r>drop 0
        ;  $1800 []key !

forth definitions

privates

: (status)      ( -- )                                  \ display statusline
        push base decimal eol                           \ save base, clear line
        get-context .vocname                            \ type context name
        #10 htab get-current .vocname                   \ type current name
        #20 htab depth 3 .r                             \ depth of the stack
        r@ 3 .r space                                   \ number base
        here .hex                                       \ dictionary pointer
        current-directory count dup #18 >               \ current path
        if      over 3 type ." ..." dup #12 - /string
        then
        type #53 htab thefile c@
        if      thefile count type #66 htab
        then
        pop base                                        \ restore base
        ;  private

internal
' (status) is status                                    \ set vector
forth

staton                                      \ show status from now on

deprive

\ include

internal also extra definitions

\G c-addr is the address of a counted string containing the path to
\G the help files.
\G See also: HELP LIBPATH
CREATE HELPPATH         ( -- c-addr )                   \ EXTRA
        here 64 dup allot erase

\G c-addr is the address of a counted string containing the path to
\G the library files.
\G See also: HELPPATH NEEDS
CREATE LIBPATH          ( -- c-addr )                   \ EXTRA
        here 64 dup allot erase

internal definitions

: mark-context
        vsp , get-current ,
        context here #vocs cells dup allot cmove
        ;

: release-context
        @+ to vsp @+ set-current
        context #vocs cells cmove
        ;

extra definitions

\G Skip leading space delimiters. Parse name delimited by a space.
\G Find name. If found continue. Otherwise, load the file with the
\G same name (excluding an optional trailing minus sign) from the
\G directory specified in LIBPATH .
: NEEDS         ( "name" -- )                           \ EXTRA
        ?undef
        if      parsed-word over c@ '-' =               \ use same word
                if      1 /string                       \ delete leading minus
                then
                libpath count temporary pack            \ path
                '\' temporary append-char               \ backslash
                append                                  \ filename
                temporary count '.' scan nip 0=     \ if needed, add extension
                if      fext$ count temporary append
                then
                temporary count included                \ load the file
        then
        ;

internal definitions

: f:marker
        release-context
        ;  compile-only

doer: domarker
        body> (forget)
        ;

' f:marker is-forget domarker

forth definitions

\G Skip leading space delimiters. Parse name delimited by a space.
\G Create a dictionary for name with the execution semantics defined
\G below.
\G
\G name Executing: ( -- )
\G Restore all dictionary allocation and search pointers to the
\G state they had just prior to the definition of name. Remove the
\G definition of name and all subsequent definitions. Restoration of
\G any structures still existing that could refer to deleted
\G definitions or deallocated data space is not necessarily
\G provided. No other contextual information such as numeric base is
\G affected.
: MARKER        ( "name" -- )                           \ FORTH
        create  mark-context domarker
        ;  ans

extra definitions

\G Display the list of words that are created by MARKER .
: .MODULES      ( -- )                                  \ EXTRA
        depth >r voc-link
        begin   regular?                                \ with headers
                if      dup voc@ temporary !
                        begin   another
                        while   dup head> >call
                                [ ' domarker >body 3 cells + ]
                                literal =
                                if      swap
                                else    drop
                                then
                        repeat
                then
                @ ?dup 0=
        until
        depth r>
        ?do     cr .head                                \ in historical order
        loop
        ;

previous forth definitions

:noname
        cr .modules r>drop 0
        ;  $4200 []key !   -- F8

\ words

extra definitions

privates

create wordbuffer       private here #32 dup allot erase
create comparebuffer    private here #32 dup allot erase

variable any?   private any? off
variable wtlr   private

\G flag is true if EVERY was typed in. Subsequent execution without
\G executing EVERY gives a false flag.
: EVERY?        ( -- flag )                             \ EXTRA "every-query"
        any? @ any? off
        ;

\G If the remaining of the current line is less than sixteen,
\G perform CR . When the cursor is not a at column dividable by 16,
\G emit spaces until the column is dividable by 16. Display the name
\G associated with the dictionary entry address dea and wait for the
\G time in milliseconds in WORDSPEED . 
: ?HEAD         ( dea -- )                              \ EXTRA "query-head"
        out #16 + c/l < invert                          \ line already too long
        if      cr
        then
        out                                             \ tab when not at start
        if      #16 out over mod - spaces
        then
        .head                                           \ type the name
        ;

: (wrd2)
        local thevoc true local new thevoc voc@ temporary !
        begin   another
        while   dup head>name hseg swap countx
                #31 and dup comparebuffer c!
                cseg comparebuffer 1+ rot cmovex
                comparebuffer count wordbuffer count search nip nip
                if      new
                        if      cr 8 spaces thevoc .vocname cr
                                clear new
                        then    wtlr incr dup ?head
                then
                drop
        repeat
        ;  private

: (wrd1)
        dup body> >head ?dup
        if      cr 8 spaces .head
        then
        cr voc@ temporary !
        begin   another
        while   ?head wtlr incr
        repeat
        ;  private

: (with)
        bl word count dup 0=
        if      voc-link
                begin   stop?
                        if      drop exit
                        then
                        regular?
                        if      dup (wrd1)
                        then
                        @ ?dup 0=
                until   exit
        then
        wordbuffer place voc-link
        begin   stop?
                if      drop exit
                then
                regular?
                if      dup (wrd2)
                then
                @ ?dup 0=
        until
        ;  private

: (words)
        every?
        if      voc-link
                begin   stop?
                        if      drop exit
                        then
                        regular?
                        if      dup (wrd1)
                        then
                        @ ?dup 0=
                until
                exit
        then
        get-context (wrd1)
        ;  private

\G Skip leading space delimiters. Parse name delimited by a space.
\G Display the words from every vocabulary containing name. Case is
\G significant.
: WITH          ( "name" -- )                           \ EXTRA
        wtlr off (with) cr ." Number of words " wtlr @ .dec
    ;

get-current get only set-current

\G List the word names in the first word list of the search order in
\G colums of 16 characters wide and a count at the end.
\G
\G WORDS is implemented using pictured numeric output words. Its use
\G will corrupt the transient region identified by #> .
\G See also: EVERY
: WORDS         ( -- )                                  \ ONLY
        wtlr off (words) cr ." Number of words " wtlr @ .dec
        ;  ans

set-current

\G Display the word lists that have a name, those who have been
\G created with VOCABULARY .
: .WORDLISTS    ( -- )                                  \ EXTRA
        cr voc-link
        begin   regular?
                if      dup .vocname space space
                then
                @ ?dup 0=
        until
        ;

\G Set a flag so that the next execution of WORDS and such words
\G will act on every vocabulary.
: EVERY         ( -- )                                  \ EXTRA
        any? on
        ;

deprive

forth definitions

\ Strings for the PROJECT library program

extra definitions

\G c-addr is the address of a counted string containing a
\G description of the project for which the file is created.
create PROJ$            ( -- c-addr )                   \ EXTRA
        here #56 dup allot erase

\G c-addr is the address of a counted string containing a
\G description of the category to which this file belongs.
create CAT$             ( -- c-addr )                   \ EXTRA
        here #56 dup allot erase

\G c-addr is the address of a counted string containing the name of
\G the creator of this file.
create CREAT$           ( -- c-addr )                   \ EXTRA
        here #56 dup allot erase

forth definitions

\ interface with DOS

extra definitions

\G A constant character that can be used as a line separator for
\G some commands, like SF DIR etc. Normally 254, ''.
#254 CONSTANT SEPARATOR     ( -- char )                 \ EXTRA

privates

create parms    private here #14 dup allot erase
create parm$    private here #64 dup allot erase

\G Execute the DOS command specified by the character string c-addr
\G u. When the screen mode or the current direcotory are changed,
\G they will be restored.
: SYSTEM        ( c-addr u -- )                         \ EXTRA
        dup                                             \ check empty string
        if      s" /c" parm$ pack append                \ keep string
                ^M parm$ count + c!     ( CP/M ? )      \ put ^m at end for DOS
        else    2drop parm$ off                         \ null string to DOS
        then
        parms #14 erase                                 \ erase array
        eseg parms !                                    \ put environment there
        parm$ parms 2 + !                               \ put offset there
        cseg parms 4 + !                                \ and segment
        comspec 1+ parms exec                           \ execute COMMAND.COM
        restore-metrics                                 \ reset cursor position
        get-directory throw                             \ get directory
        current-directory place                 \ keep status line up to date
        ;

\G Skip leading space delimiters. Parse name delimited by a space.
\G Create a dictionary entry for name with the execution semantics
\G defined below.
\G
\G name Executing: ( "ccc" -- )
\G Execute the DOS command specified by the character string c-addr
\G u and parameters ccc, terminated by the end of the line or the
\G character in SEPARATOR .
: DOS:          ( c-addr u "name" -- )                  \ EXTRA "dos-colon"
        create  here pack c@ 1+ allot                   \ compile dos name
        does>   count temporary pack >r                 \ copy string
                separator word count                    \ get any parameters
                bl skip -trailing                       \ discard spaces
                dup
                if      bl r@ append-char               \ insert a space
                        r@ append                       \ append parameters
                else    2drop                           \ no parameters
                then
                r> count cr system                      \ and call DOS
        ;

\G Skip leading space delimiters. Parse ccc delimited by a space.
\G When ccc is the null string, display the current directory. Else
\G change to the directory ccc. Contrary to DOS, when a drive letter
\G and a colon are in front of the string, that drive will also be
\G made current.
: CD            ( "ccc" -- )                            \ EXTRA "change-dir"
        bl word c@                                      \ when no empty string
        if      here count set-directory throw          \ set current directory
        then
        get-directory throw current-directory place     \ keep in a safe place
        cr ." Directory: "                              \ like iForth
        current-directory count type space              \ and inform user
        ;

\G Skip leading space delimiters. Parse name delimited by a space.
\G Create a dictionary entry for name with the execution semantics
\G defined below.
\G
\G name Executing: ( -- )
\G Change the default drive number to n, n is zero for drive A:.
: DRIVE         ( n "name" -- )                         \ EXTRA
        create  c,
        does>   c@ setdisk drop                         \ select drive
                get-directory throw                     \ get current directory
                current-directory place                 \ and update status
        ;

0 drive a:  1 drive b:  2 drive c:  3 drive d:  4 drive e:
5 drive f:  6 drive g:  7 drive h:  8 drive i:  9 drive j:

deprive

forth definitions

                            \ (* End of Source *) /

