\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : Meta compiler for CHForth version 1.1.0 
\ FILENAME    : FILES.FRT
\ DESCRIPTION : High level reading and writing ASCII files 
\ AUTHOR      : Coos Haak, Utrecht 
\ CREATED     : May 11, 1994
\ ----------------------------------------------------------------------



\G x is the value for selecting the "read-only" file access method.
\G See also: CREATE-FILE OPEN-FILE
0 CONSTANT R/O  ANS ( -- x )                    \ FORTH "r-o"

\G x is the value for selecting the "write-only" file access method.
\G See also: CREATE-FILE OPEN-FILE
1 CONSTANT W/O  ANS ( -- x )                    \ FORTH "w-o"

\G x is the value for selecting the "read-write" file access method.
\G See also: CREATE-FILE OPEN-FILE
2 CONSTANT R/W  ANS ( -- x )                    \ FORTH "r-w"

\G Modify the file access method x1 to additionally select a
\G "binary", i.e. not line oriented, file access method, giving
\G access method x2.
\G See also: R/O R/W W/O
CODE BIN        ( x1 -- x2 )                    \ FORTH
                NEXT
END-CODE  ANS

\G Create the file named in the character string specified by c-addr
\G and u, and open it with file access method x1. If a file with the
\G same name already exists, recreate it as an empty file.
\G
\G If the file was successfully created and opened, ior is zero, x2
\G is the fileid, and the file has been positioned at the start of
\G the file.
\G
\G Otherwise ior is the I/O result code and x2 is an unspecified
\G value.
: CREATE-FILE   ( c-addr u x1 -- x2 ior )       \ FORTH
        PUSH SOURCE-ID (CREATE-FILE) POP SOURCE-ID
        ;  ANS

\G Open the file named in the character string specified by c-addr
\G u, with file access indicated by x1.
\G
\G If the file was successfully opened, ior is zero, x2 is the
\G fileid, and the file has been positioned at the start of the
\G file.
\G
\G Otherwise ior is the I/O result code and x2 is an unspecified
\G value.
: OPEN-FILE     ( c-addr u x1 -- x2 ior )       \ FORTH
        PUSH SOURCE-ID (OPEN-FILE) POP SOURCE-ID
        ;  ANS

\G Close the file identified by fileid, ior is the I/O result code.
: CLOSE-FILE    ( fileid -- ior )               \ FORTH
        PUSH SOURCE-ID (CLOSE-FILE) POP SOURCE-ID
        ;  ANS

\G Read the next line from the file specified by fileid into memory
\G at the address c-addr. At most u1 characters are read. Up to two
\G line terminating characters may be read into memory at the end of
\G the line, but are not included in the count u2. The line buffer
\G provided by c-addr should be at least u1+2 characters long.
\G
\G If the operation succeeded, flag is true and ior is zero. If a
\G line terminator was received before u1 characters were read, then
\G u2 is the number of characters, not including the line
\G terminator, actually read (0 <= u2 <= u1). When u1 = u2 the line
\G terminator has yet to be reached.
\G
\G If the operation is initiated when the value returned by
\G FILE-POSITION is equal to the value returned by FILE-SIZE for the
\G file identified by fileid, flag is false, ior is zero, and u2 is
\G zero. If ior is non-zero, an exception occurred during the
\G operation and ior is the I/O result code.
\G
\G **** Veranderen
\G An ambiguous condition exists if the operation is initiated when
\G the value returned by FILE-POSITION is greater than the value
\G returned by FILE-SIZE for the file identified by fileid, or the
\G requested operation attempts to read portions of the file not yet
\G written.
\G
\G At the conclusion of the operation, FILE-POSITION returns a value
\G past the characters consumed by the operation.
: READ-LINE     ( c-addr u1 fileid -- u2 flag ior )         \ FORTH
        PUSH SOURCE-ID (READ-LINE) POP SOURCE-ID
        ;  ANS

\G Write u characters from c-addr followed by the line terminators
\G to the file identified by fileid starting at its current
\G position. ior is the I/O result code.
\G
\G At the conclusion of the operation, FILE-POSITION returns a value
\G past the characters written to the file and FILE-SIZE returns a
\G value greater then or equal to the value returned by
\G FILE-POSITION .
\G See also: READ-FILE READ-LINE
: WRITE-LINE    ( c-addr u fileid -- ior )              \ FORTH
        DUP>R WRITE-FILE ?DUP
        IF      R>DROP EXIT
        THEN
        (LIT) _crlf 2 R> WRITE-FILE
        ;  ANS

\G Read u1 consecutive characters to c-addr from the current
\G position of the file identified by fileid.
\G
\G If u1 characters are read without an exception, ior is zero and
\G u2 is equal to u1.
\G
\G If the end of the file is reached before u1 characters are read,
\G ior is zero and u2 is the number of characters actually read.
\G
\G If the operation is initiated when the value of FILE-POSITION is
\G equal to the value returned by FILE-SIZE for the file identified
\G by fileid, ior is zero and u2 is zero.
\G
\G If an exception occurs, ior is the I/O result code and u2 is the
\G number of characters transferred to c-addr without an exception.
\G
\G An ambiguous condition exists if the operation is initiated when
\G the value returned by FILE-POSITION is greater than the value
\G returned by FILE-SIZE for the file identified by fileid, or if
\G the requested operation attempts to read portions of the file not
\G written.
\G
\G At the conclusion of the operation FILE-POSITION returns a value
\G past the characters consumed by the operation.
CODE READ-FILE  ( c-addr u1 fileid -- u2 ior )          \ FORTH
                POP     CX
                POP     DX
                MOV     AH, # 3F
                INT     21
                SBB     BX, BX
        U< IF
                MOV     BL, AL
                MOV     BH, # FE
                XOR     AX, AX              \ Unknown number of chars read
        THEN
                PUSH    AX
                NEXT
END-CODE  ANS

\G Write u characters from c-addr to the file identified by fileid
\G starting at its current position. ior is the I/O result code.
\G
\G At the conclusion of the operation FILE-POSITION returns a value
\G past the characters written to the file and FILE-SIZE returns a
\G value greater than or equal to the value returned by
\G FILE-POSITION .
\G See also: READ-FILE WRITE-LINE
CODE WRITE-FILE         ( c-addr u fileid -- ior )      \ FORTH
                POP     CX
                POP     DX
                MOV     AH, # 40
                INT     21
                JB      E $
                XOR     BX, BX
                CMP     AX, CX
                JZ      F $
                MOV     BX, # #-37
                JMP     F $
        E $:    MOV     BL, AL
                MOV     BH, # FE
        F $:    NEXT
END-CODE  ANS

\G Delete the file named in the character string specified by c-addr
\G u. ior is the I/O result code.
: DELETE-FILE   ( c-addr u -- ior )                     \ FORTH
        (NAME) FDEL
        ;  ANS

\G ud is the current file position for the file identified by
\G fileid. ior is the I/O result code.
: FILE-POSITION     ( fileid -- d ior )                 \ FORTH
        >R 0 0 1 R> MOVE-FILE-POINTER
        ;  ANS

\G Reposition the file identified by fileid to ud. ior is the I/O
\G result code. An ambiguous condition exists if the file is
\G positioned outside the file boundaries.
: REPOSITION-FILE   ( ud fileid -- ior )                \ FORTH
        0 SWAP MOVE-FILE-POINTER NIP NIP
        ;  ANS

\G ud is the size, in characters, of the file identified by
\G fileid. ior is the I/O result code. This operation does not
\G effect the value returned by FILE-POSITION .
: FILE-SIZE         ( fileid -- ud ior )                \ FORTH
        >R 0 0 1 R@ MOVE-FILE-POINTER ?DUP
        IF      R>DROP EXIT                                     \ error
        THEN
        0 0 2 R@ MOVE-FILE-POINTER ?DUP
        IF      R>DROP NIP NIP EXIT                             \ error
        THEN
        2SWAP 0 R> MOVE-FILE-POINTER NIP NIP                    \ fine?
        ;  ANS

\G Set the size of the file identified by fileid to ud. ior is
\G the I/O result code.
\G
\G If the resultant file is larger than the file before the
\G operation, the portion of the file added as a result of the
\G operation may not have been written. 
\G
\G At the conclusion of the operation FILE-SIZE returns the value
\G ud and FILE-POSITION returns an unspecified value.
\G See also: READ-FILE READ-LINE
: RESIZE-FILE       ( ud fileid -- ior )                \ FORTH
        DUP>R REPOSITION-FILE ?DUP
        IF      R>DROP EXIT                                     \ error
        THEN
        0 0 R> WRITE-FILE                                       \ fine?
        ;  ANS

\G Return the status of the file identified by the character
\G string c-addr u. If the file exists, ior is zero; otherwise
\G ior is the I/O result code. x contains the DOS attribute of
\G the file.
: FILE-STATUS       ( c-addr u -- x ior )               \ FORTH
        (NAME) FEXIST                                   \ x is DOS attribute
        ;  ANS

\G Attempt to force any buffered information written to the file
\G referred to by fileid to be written to mass storage, and the
\G size information for the file to be recorde in the storage
\G directory if changed. If the operation is successfull, ior is
\G zero. Otherwise ior is the I/O result code.
: FLUSH-FILE        ( fileid -- ior )                   \ FORTH
        NOT-IMPLEMENTED                                 \ what's the use?
        ;  ANS

\G Rename the file named by character string c-addr1 u1 to the
\G name in the character string c-addr2 u2. ior is the I/O result
\G code.
: RENAME-FILE   ( c-addr1 u1 c-addr2 u2 -- ior )        \ FORTH
        (NAME) -ROT (LIT) T[ filename 40 + ,-L T]
        DUP 40 ERASE PACK 1+ SWAP FRENAME
        ;  ANS

\G Remove fileid from the stack. Save the current input source
\G specification, including the current value of SOURCE-ID .
\G Store fileid in SOURCE-ID . Make the file specified by fileid
\G the input source. Store zero in BLK . Other stack effects are
\G due to the words INCLUDEd.
\G
\G Repeat until end of file: read a line from the file, fill the
\G input buffer from the contents of that line, set >IN to zero,
\G and interpret.
\G
\G Interpretation begins at the file position where the next file
\G read would occur.
\G
\G When the end of the file is reached, close the file and
\G restore the input source specification to its saved value.
\G
\G An ambiguous condition exists if fileid is invalid, if an I/O
\G exception occurs reading fileid, or an I/O exception occurs
\G while closing fileid. When an ambiguous condition exists, the
\G status (open or closed) of any files that were being
\G interpreted is implementation defined.
: INCLUDE-FILE  ( fileid -- )                           \ FORTH
        ERRNAME OFF ERRLINE OFF                         \ Assume no errors
        SAVE-INPUT                                      \ Save input specs
        >R 2>R 2>R 2>R                                  \ Secret information
        TO SOURCE-ID                                    \ New file id
        BLK OFF IN-USE ON #LINES OFF                    \ No blocks, line zero
        .STATUS                                         \ Display status line
        BEGIN   REFILL                                  \ Read a line
        WHILE   ECHO?
                IF      CR SOURCE TYPE                  \ If needed, display
                THEN
                INTERPRET                               \ Interpret it
                LINESREAD INCR                          \ Next line number
        REPEAT
        .STATUS                                         \ Display status line
        SOURCE-ID CLOSE-FILE THROW                      \ Close the file
        2R> 2R> 2R> R>                                  \ Secret information
        RESTORE-INPUT THROW                             \ Restore input specs
        ;  ANS

\G Remove c-addr u from the stack. Save the current input source
\G specification, including the current value of SOURCE-ID . Open
\G the file specified by c-addr u, store the resulting fileid in
\G SOURCE-ID and make it the input source.  Store zero in BLK .
\G Other stack effects are due to the words INCLUDEd.
\G
\G Repeat until end of file: read a line from the file, fill the
\G input buffer from the contents of that line, set >IN to zero,
\G and interpret.
\G
\G Interpretation begins at the file position where the next file
\G read would occur.
\G
\G When the end of the file is reached, close the file and
\G restore the input source specification to its saved value.
\G
\G An ambiguous condition exists if the named file can not be
\G opened, if an I/O exception occurs reading the file, or an I/O
\G exception occurs closing the file. When an ambiguous condition
\G exists, the status (open or closed) of any files that were
\G being interpreted is implementation defined.
\G See also: INCLUDE-FILE
: INCLUDED      ( c-addr u -- )                 \ FORTH
        R/O OPEN-FILE THROW INCLUDE-FILE
        ;  ANS

EXTRA:

:ORPHAN OPEN/CREATE     ( fid ior -- fid ior )
        DUP
        IF      EXIT                                    \ error
        THEN
        DROP TO SOURCE-ID                               \ keep fid
        (LIT) filename COUNT 'NAME PLACE                \ keep name
        IN-USE OFF
        #LINES OFF
        SOURCE-ID FALSE
        ;

:ORPHAN (CREATE-FILE)     ( c-addr u fam -- fid ior )
        DROP                                            \ ignore fam
        (NAME)
        0 FCREATE
        OPEN/CREATE
        ;

:ORPHAN (OPEN-FILE)     ( c-addr u fam -- fid ior )
        >R (NAME) R>
        FOPEN
        OPEN/CREATE
        ;

LABEL rdlen     0 ,
LABEL rdadr     0 ,
LABEL rdcnt     0 ,
LABEL rdsiz     0 , 0 ,
LABEL rdptr     0 , 0 ,

:ORPHAN (READ-LINE)     ( c-addr u1 fid -- u2 flag ior )
        TO SOURCE-ID (LIT) rdlen 2!                     \ save parameters

        SOURCE-ID FILE-POSITION DUP
        IF      EXIT                                    \ error
        THEN
        DROP (LIT) rdptr 2!

        0 0 2 SOURCE-ID MOVE-FILE-POINTER DUP
        IF      EXIT                                    \ error
        THEN
        DROP (LIT) rdsiz 2!

        (LIT) rdptr 2@ SOURCE-ID REPOSITION-FILE DUP
        IF      FALSE FALSE ROT EXIT                    \ error
        THEN
        DROP

        (LIT) rdptr 2@ (LIT) rdsiz 2@ DU< INVERT
        IF      FALSE FALSE FALSE EXIT                  \ end of file
        THEN

        (LIT) rdlen 2@ SOURCE-ID READ-FILE DUP
        IF      FALSE SWAP EXIT                         \ error
        THEN
        DROP

        (LIT) rdadr @ SWAP ^J SCAN DROP                 \ first lf char
        DUP 1+ (LIT) rdadr @ - (LIT) rdlen !            \ calc eol+cr+lf
        DUP 1- C@ DUP ^M = SWAP ^Z = OR +               \ prev char cr or eof
        (LIT) rdadr @ - (LIT) rdcnt !                   \ calc eol-cr-eof

        (LIT) rdptr 2@ (LIT) rdlen @ M+
        SOURCE-ID REPOSITION-FILE DUP
        IF      FALSE FALSE ROT EXIT                    \ error
        THEN
        DROP

        (LIT) rdcnt @ TRUE FALSE                        \ ok
        ;

:ORPHAN (CLOSE-FILE)    ( fid -- ior )
        TO SOURCE-ID IN-USE OFF
        SOURCE-ID FCLOSE
        ;

INTERNAL:

LABEL handles   THANDLES T/HANDLE * ALLOT           \ Handle stack

\ Return the base address of the handle stack.
handles CONSTANT HANDLE-0       ( -- addr )         \ EXTRA "handle-zero"

\ n is the size in characters of a file handle.
T/HANDLE CONSTANT /HANDLE       ( -- n )            \ EXTRA "per-handle"

\ n is the maximum number of usable handles.
THANDLES CONSTANT HANDLES       ( -- n )            \ EXTRA

\ Index into the current fid structure.
CODE []HANDLE           ( x -- addr )           \ EXTRA "handle-array"
                MOV     AX, # T/HANDLE
                MUL     T' SOURCE-ID
                ADD     AX, T' HANDLE-0
                ADD     BX, BX
                ADD     BX, AX
                NEXT
END-CODE

: IN-USE                ( -- addr )             \ file is used by
        0 []HANDLE                              \ INCLUDE-FILE
        ;

EXTRA:

\G A variable containing the number of the current line of the
\G current file.
: #LINES                ( -- addr )             \ EXTRA "number-lines"
        1 []HANDLE
        ;

\G Contains the name of the current file.
: 'NAME                 ( -- addr )             \ EXTRA "tick-name"
        2 []HANDLE
        ;

INTERNAL:

\ Close all files opened of reading. Used in exception handlers.
: CLOSE-ALL-FILES       ( -- )                  \ EXTRA
        1 HANDLES 1-
        DO      I TO SOURCE-ID IN-USE @
                IF      CR ." Closing " 'NAME COUNT TYPE
                        SOURCE-ID CLOSE-FILE DROP
                THEN
        -1 +LOOP
        CLEAR SOURCE-ID
        ;

\ Reposition the file pointer to line x of the current fid. If
\ the operation fails, ior contains an exception number.
: REPOSITION-LINE       ( x -- ior )            \ EXTRA
        0 0 SOURCE-ID REPOSITION-FILE ?DUP
        IF      NIP EXIT
        THEN
        #LINES OFF
        >IN @ SWAP
        BEGIN   REFILL
        WHILE   #LINES @ OF     >IN ! FALSE EXIT
                THEN
        REPEAT
        2DROP FE1E
        ;

FORTH:

                            \ (* End of Source *) /
