\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : A Forth Assertions Construct
\ CATEGORY    : Example
\ AUTHOR      : M. Anton Ertl
\ LAST CHANGE : July 17, 1994, Coos Haak 
\ LAST CHANGE : July 13, 1994, Marcel Hendrix 
\ ----------------------------------------------------------------------



        MARKER -assert


        PRIVATES


DOC
From: anton@mips.complang.tuwien.ac.at (Anton Ertl)
Subject: Assertions
Date: 27 Jun 1994 10:13:18 GMT
  
I have been thinking about adding assertions (in the sense of e.g. C's
assert.h) to Forth. Has anybody done this and what is your syntax?
  
My idea was to have a word
  
assert ( f -- )
  
However, this is not so nice since we want to be able to switch off
assertions completely (for performance reasons). So my current
assertions syntax is:
  
assert( code generating a flag )
  
This THROWs if the flag is not true. The assertion consumes the flag.
E.g.,
  
50 0 ?do
  assert( i 50 < ) ( usually the conditions are not so obviously true)
                  
loop
  
If assertions are turned off, there "assert(" acts just like "(",
i.e., the assertion is treated as comment. One disadvantage of this
syntax is that you cannot have comments in assertions.
  
Since usually we do not want to switch all assertions on or off, I'm thinking 
about introducing four levels of assertions.
  
assert1(
Your typical assertion. Switched on by default. "assert(" is an alias
for "assert1("
  
assert2(
An assertion only turned on for debugging.
  
assert0(
An assertion that's so essential (and/or fast) that you want it turned on even 
when you turn off "assert1("s.
  
assert3(
An assertion that's so time-consuming that you do not usually want it
turned on even for normal debugging. You would turn it on for, e.g.,
overnight test runs.
  
What do you think?
  
- anton
-- 
M. Anton Ertl                    Some things have to be seen to be believed
anton@mips.complang.tuwien.ac.at Most things have to be believed to be seen
  
From: anton@mips.complang.tuwien.ac.at (Anton Ertl)
Subject: Re: Assertions
Date: 11 Jul 1994 13:21:07 GMT
  
Only two persons answered to my query: Phil Koopman told me about his
conditional compilation word T\, and Raul Deluth Miller pointed me to
ABORT" (which does not quite do the job).
  
In the meantime I have implemented my proposal.
  
In article <2vgjndINN211@minnie.informatik.uni-kiel.de>, uho@informatik.uni-kie
l.de (Ulrich Hoffmann) writes:
|> ASSERT should be able to output useful messages if the assertion
|> fails. So my signature of assert is:
|> 
|> : assert ( flag addr len -- ) \ signal error if assertion failed
  
My assert( displays its file and line number (in a format compatible
with emacs' compile-mode, so I need only three keypresses to get to
the source).
  
Adding a message string is probably overkill. Since assertions are
there for catching programming errors, the user cannot do anything
about them (and may have no idea what the message means). And for the
programmers the source code position offers enough
information. Besides, they don't want to waste their time in error
messages that will hopefully never be used.
  
Your assert reminds me very much of ANS Forth's ABORT", which is
appropriate for handling user errors.
  
|> >Since usually we do not want to switch all assertions on or off, I'm
|> >thinking about introducing four levels of assertions.
|> 
|> Hmm. Looks too complex for me.  I would add  ASSERTIONS ON  or
|> ASSERTIONS OFF  directives to my code according to the parts of the
|> program I would like to be checked.
  
I would have to have up to one ASSERTIONS ON or OFF per
assertion. Even worse, changing the amount of checking would require a
lot of editing. I think in practice programmers would forget about
that and just turn them all on or off, no matter what they check and
what they cost. Also having four levels is not very complex. It just
cost a few lines:
   
variable assert-level \ all assertions above this level are turned off
1 assert-level !
   
: assertn ( n -- ) \ the real work is done by )
    assert-level @ >
    if
        POSTPONE (
    then ;
  
: assert0( ( -- )
    0 assertn ; immediate
: assert1( ( -- )
    1 assertn ; immediate
: assert2( ( -- )
    2 assertn ; immediate
: assert3( ( -- )
    3 assertn ; immediate
: assert( ( -- )
    POSTPONE assert1( ; immediate
  
I don't show you my ")", since getting and storing the source position
are done in a system-dependent way, but a simple, portable ")" would
look like this:
  
: ) ( f -- )
 0= abort" assertion failed" ;
  
- anton
-- 
M. Anton Ertl                    Some things have to be seen to be believed
anton@mips.complang.tuwien.ac.at Most things have to be believed to be seen
ENDDOC

\G A value. All assertions above this value are turned off.
\G Default value is one.
1 VALUE ASSERT-LEVEL    ( -- x )                \ ASSERT  "assert-level"

-- The real work is done by a routine with the name ")".
: ASSERTN                       \ <n> --- <>
        assert-level >
        IF      postpone (
        THEN
        ;  PRIVATE

\G An assertion at level 0. Always enabled, unless assert-level
\G is negative.
: ASSERT0(      ( -- )                          \ ASSERT  "assert-zero"
        0 assertn
        ;  IMMEDIATE  COMPILE-ONLY

\G An assertion at level 1. Enabled when assert-level is 1 or
\G higher.
: ASSERT1(      ( -- )                          \ ASSERT  "assert-one"
        1 assertn
        ;  IMMEDIATE  COMPILE-ONLY

\G An assertion at level 2. Enabled when assert-level is two or
\G higher.
: ASSERT2(      ( -- )                          \ ASSERT  "assert-two"
        2 assertn
        ;  IMMEDIATE  COMPILE-ONLY

\G An assertion at level 3. Enabled when assert-level is three or
\G higher.
: ASSERT3(      ( -- )                          \ ASSERT  "assert-three"
        3 assertn
        ;  IMMEDIATE  COMPILE-ONLY

\G An assertion at level 1. Equivalent to ASSERT1( .
: ASSERT(       ( -- )                          \ ASSERT  "assert"
        POSTPONE assert1(
        ;  IMMEDIATE  COMPILE-ONLY

#-622 MESS" assertion failed"

INTERNAL

: do-)          \ <flag> <c-addr> <u> <#lines> -- <>
        3 PICK
        IF      2DROP 2DROP EXIT
        THEN
        #LINES ! 'NAME PLACE
        'NAME ERRNAME ! #LINES @ ERRLINE !
        SOURCE-ID 0=
        IF      1 TO SOURCE-ID          \ force logging
                #-622 FINDMESSAGE DROP LOG-ERROR
                CLEAR SOURCE-ID
        THEN
        #-622 THROW
        ;  PRIVATE

FORTH

\G The action of this word depends on a preceding ASSERTx( word.
\G When assert-level is higher than x, the text ccc is just a
\G comment. When assert-level is lower than x, ccc is compiled,
\G followed by the name of the current source file and line
\G number. When at run-time the code ccc evaluates to zero, the
\G source file name and line number are printed and an exception
\G is thrown. The action of this routine is similar to an ABORT"
\G that fires at compile-time.
: )             ( "ccc" -- )                    \ ASSERT  "right-paren"
        SOURCE-ID 0> INVERT ABORT" assertion does not work from the terminal"
        'NAME COUNT POSTPONE SLITERAL
        #LINES @ POSTPONE LITERAL
        POSTPONE do-)
        ;  IMMEDIATE  COMPILE-ONLY

: .ASSERT
        CR ."  x TO assert-level -- set assertion level to x (at compile-time)"
        CR ."  ASSERTx( cccc )   -- abort if cccc evaluates to false _and_ assert-level"
        CR ."                       greater than or equal to x" 
        CR ."  ASSERT( cccc )    -- equivalent to ASSERT1( ccc )" 
        CR
        CR ." If assert-level is >= 3 all assertions are in."
        CR ." If assert-level is 0 assert0 is still in." 
        CR ." If assert-level is 0< all asserts are out."
        ;
        
DEPRIVE

                            \ (* End of Source *) /
