;*******************************;
; WASM Tick Timer               ;
; By Eric Tauck                 ;
;                               ;
; Defines:                      ;
;                               ;
;   TicSys  current system time ;
;   TicRes  reset a timer       ;
;   TicPas  return ticks passed ;
;   TicWai  pause for a time    ;
;*******************************;

;--- get current time

        call    TicSys                          ;get current time
        sub     bx, bx                          ;zero offset
        mov     cx, TICK_TIMERS                 ;number of timers

;--- initialize each timer

_ticks1 mov     [_ticks_times + bx], ax         ;save low word
        mov     [_ticks_times + bx + 2], dx     ;save high word
        add     bx, 4                           ;point to next entry
        loop    _ticks1                         ;loop for each timer

        jmp     _ticks_end

;--- equates

_TIMER_MAX1  EQU     0018H   ;ticks per day, high word
_TIMER_MAX2  EQU     00B0H   ;ticks per day, low word

;--- timer data

TICK_TIMERS     EQU     10              ;number of timers (at least one)
_ticks_times    DS      TICK_TIMERS * 4 ;timer values

;========================================
; Return the system time in ticks. Note:
; the time is read directly because using
; the BIOS resets the overflow flag
; (which keeps DOS from detecting a new
; day).
;
; Out: DX:AX= tick value.

TicSys  PROC    NEAR
        push    ds
        mov     ax, 40H         ;BIOS data segment
        mov     ds, ax
        mov     ax, [6CH]       ;load low word
        mov     dx, [6EH]       ;load high word
        pop     ds
        ret
        ENDP

;========================================
; Zero (reset) a timer value.
;
; In: AL= timer number.

TicRes  PROC    NEAR
        sub     ah, ah          ;zero high byte
        shl     ax
        shl     ax              ;index times four

        push    ax
        call    TicSys          ;get current ticks
        pop     bx

        mov     [_ticks_times + bx], ax         ;save low word
        mov     [_ticks_times + bx + 2], dx     ;save high word
        ret
        ENDP

;========================================
; Return the time passed.
;
; In: AL= timer number.
;
; Out: DX:AX= ticks passed.

TicPas  PROC    NEAR
        push    si
        sub     ah, ah          ;zero high byte
        shl     ax
        shl     ax              ;index times four

        push    ax
        call    TicSys          ;get current time
        pop     si

        cmp     dx, [_ticks_times + si + 2]     ;compare high word
        ja      _tcpas2                         ;jump if no wrap
        jb      _tcpas1                         ;jump if wrap
        cmp     ax, [_ticks_times + si]         ;compare low word
        jae     _tcpas2                         ;jump if no wrap

;--- wrap

_tcpas1 mov     bx, _TIMER_MAX1                 ;ticks per day in BX:CX
        mov     cx, _TIMER_MAX2                 ;
        sub     cx, [_ticks_times + si]         ;wrap around around ticks
        sbb     bx, [_ticks_times + si + 2]     ;
        add     ax, cx                          ;add current ticks
        adc     dx, bx                          ;
        pop     si
        ret

;--- no wrap

_tcpas2 sub     ax, [_ticks_times + si]         ;calculate ticks passed
        sbb     dx, [_ticks_times + si + 2]     ;
        pop     si
        ret
        ENDP

;========================================
; Wait for a period of time.
;
; In: AX= ticks to pause.

TicWai  PROC    NEAR
        push    di
        mov     di, ax

        push    WORD [_ticks_times]     ;
        push    WORD [_ticks_times + 2] ;save timer 0

        sub     al, al                  ;timer zero
        call    TicRes                  ;reset timer
_tcwai1 sub     al, al                  ;timer zero
        call    TicPas                  ;get time passed
        cmp     ax, di                  ;check if timeout
        jbe     _tcwai1                 ;loop back if not time yet

        pop     WORD [_ticks_times + 2] ;
        pop     WORD [_ticks_times]     ;restore timer 0
        pop     di
        ret
        ENDP

_ticks_end
